#
#  NotificationCenter.rb
#  Allows objects to register to receive event notifications that other objects
#  can then post.
#  
#  Created by Kevin Bullock on Sun Nov 30 2003.
#  Copyright (c) 2003 Kevin Bullock.
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#  
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#  
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
#  You may also contact the author of this program at <kbullock@ringworld.org>.
#

=begin

= NotificationCenter

== Overview

This class enables an object to broadcast a notification that an event occurred
to other objects, without having any knowledge of the objects that it is
broadcasting to. An object posts a notification (identified by a string and
possibly an associated object) to a NotificationCenter (generally the default
one). The NotificationCenter then takes care of sending the notification to all
of the objects that have registered to receive it.

The purpose of the optional associated object is generally to pass the object
posting the notification to the receivers, so that the receivers can query the
posting object for more information about the event.

Multiple notification centers may be created (as usual, with the
((<(({new}))|NotificationCenter.new>)) method), but there is one
NotificationCenter available by default through the method
(({((<NotificationCenter.instance>))})). This is a weak form of the Singleton
design pattern.

The NotificationCenter is actually a combination of three
((<design patterns|URL:http://www.csc.calpoly.edu/~dbutler/tutorials/winter96/patterns/>)):
the Singleton, Mediator, and Observer patterns. It is based closely on the
((<NSNotificationCenter|URL:http://developer.apple.com/documentation/Cocoa/Reference/Foundation/ObjC_classic/Classes/NSNotificationCenter.html>))
included in ((<Mac OS X|URL:http://www.apple.com/macosx/>))'s Cocoa framework.

== Usage

A short example of the usage of a NotificationCenter:

    require 'NotificationCenter'
    
    class Foo
        def foo
            NotificationCenter.instance.post('randomNotification', self)
        end
    end
    
    class Bar
        def initialize
            NotificationCenter.instance.add_observer(self, :notified,
                'randomNotification', nil)
        end
        
        def notified(object)
            puts 'Got a randomNotification with object ' + object.inspect
        end
    end
    
    foo = Foo.new
    bar = Bar.new
    5.times do |i|
        sleep(rand(5))
        foo.foo
    end

Here, Foo defines a method foo that posts a notification to the default
NotificationCenter. Bar registers to receive that notification, and defines
a method to be called when it does receive it. We then create a new instance
of each class, then call Foo#foo five times, sleeping a random amount of time
between each call.

The output is something like the following:

    Got a randomNotification with object #<Foo:0xc388c>
    Got a randomNotification with object #<Foo:0xc388c>
    Got a randomNotification with object #<Foo:0xc388c>
    Got a randomNotification with object #<Foo:0xc388c>
    Got a randomNotification with object #<Foo:0xc388c>

=== Class methods

--- NotificationCenter.instance
    Returns the default NotificationCenter. There is one NotificationCenter by
    default available to a program, created the first time this method is
    called. You may also create other instances of NotificationCenter if you so
    desire.
    
--- NotificationCenter.new
    Returns a new NotificationCenter that you may post notifications to and
    register observers with.

=== Instance methods

--- NotificationCenter#add_observer(observer, selector, name, object)
    Registers the object ((|observer|)) to receive notifications identified by
    the string ((|name|)) and, optionally, the object ((|object|)). When such
    a notification is posted, the NotificationCenter will invoke the method of
    ((|observer|)) indicated by ((|selector|)). If ((|object|)) is (({nil})),
    then the observer will be sent ((*all*)) notifications named ((|name|)).

--- NotificationCenter#delete_observer(observer)
    Removes ((*all*)) registrations made by ((|observer|)). Be sure to call this
    method before ((|observer|)) is destroyed, so that the NotificationCenter
    doesn't try to notify (({nil})).

--- NotificationCenter#post(name, object)
    Posts a notification identified by the string ((|name|)) and, optionally,
    the object ((|object|)) to the NotificationCenter. Note that this is a
    blocking call, that is, it does not return until notification has been sent
    to all registered observers.

== To do

* Rework ((<(({NotificationCenter#add_observer}))>)) to take a block instead of
  ((|selector|))

* Allow ((<(({NotificationCenter#delete_observer}))>)) to remove registration
  for a single notification instead of all at once

* Make sure the NotificationCenter is thread-safe.

=end

# Encapsulates a registration of an observer
class Observation
    def initialize(observer, selector, name, object)
        unless observer.respond_to? selector
            raise NameError, 'Observer must respond to ' + selector.to_s
        end
        unless observer.method(selector).arity == 1
            raise NameError, observer.class.to_s + '#' + selector.to_s + ' must take exactly one argument'
        end
        
        @observer = observer
        @selector = selector
        @name = name
        @object = object
    end
    
    def matches?(observer)
        true if @observer.equal? observer
    end
    
    def observing?(name, object)
        true if @name == name and (@object.equal? object or @object.nil?)
    end
    
    def send(object)
        @observer.send @selector, object
    end
end

class NotificationCenter
    @@default_center = nil
    
    def initialize
        @observations = Array.new
    end
    
    def NotificationCenter.instance
        begin
            Thread.critical = true
            @@default_center ||= NotificationCenter.new
        ensure
            Thread.critical = false
        end
        @@default_center
    end
    
    def post(name, object)
        @observations.each do | obs |
            if obs.observing?(name, object)
                obs.send(object)
            end
        end
    end
    
    def add_observer(observer, selector, name, object)
        @observations.push(Observation.new(observer, selector, name, object))
    end
    
    def delete_observer(observer)
        @observations.delete_if { |obs| obs.matches? observer }
    end
end
