// -*- mode: c++; indent-tabs-mode: nil -*-
//! @file SoapClient.qm SOAP Client module implementation based on the WSDL classes

/*  SoapClient.qm Copyright (C) 2012 - 2023 Qore Technologies, s.r.o.

    Permission is hereby granted, free of charge, to any person obtaining a
    copy of this software and associated documentation files (the "Software"),
    to deal in the Software without restriction, including without limitation
    the rights to use, copy, modify, merge, publish, distribute, sublicense,
    and/or sell copies of the Software, and to permit persons to whom the
    Software is furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in
    all copies or substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
    DEALINGS IN THE SOFTWARE.
*/

// make sure we have the required qore version


// do not use $ for vars

// do not ignore argument errors

// require type declarations

// enable all warnings


/** @mainpage SoapClient Module

    @tableofcontents

    @section soapclientintro Introduction to the SoapClient Module

    The %SoapClient module implements a SOAP client for %Qore using WSDL, XSD, and SOAP support implemented in the
    <a href="../../WSDL/html/index.html">WSDL</a> module.

    Create outgoing request messages by setting up a Qore data structure corresponding to the SOAP message.
    Exceptions will be thrown if either the outgoing or the response message do not corespond to the WSDL.  The
    exceptions should be fairly verbose to allow you to quickly correct any mistakes.

    This module provides the following classes:
    - @ref SoapClient::SoapClient "SoapClient"
    - @ref SoapClient::SoapConnection "SoapConnection"

    This module requires the following modules which are also reexported to the program context:
    - <a href="../../xml/html/index.html">xml</a> binary module
    - <a href="../../WSDL/html/index.html">WSDL</a> user module
    - @ref mimeintro "Mime user module"

    @section Example

    Here is an example of how to use this module:
    @code{.py}
%requires SoapClient
SoapClient sc = new SoapClient({"wsdl": "http://soap.server.org:8080/my-service?wsdl"});
hash<auto> msg = getMessage();
auto result = sc.call("SubmitDocument", msg);
    @endcode

    The SoapClient::constructor() takes named arguments in the form of a hash; valid arguments are:
    - required keys: one of:
      - \c "wsdl" or \c "wsdl_file": a string defining the WSDL or the URL of the WSDL
    - optional keys:
      - \c "service": the name of the "service" to use (if more than 1 service is defined in the WSDL then this key is
        mandatory)
      - \c "port": in case multiple port entries are found in the WSDL, give the one to be used here
      - \c "url": to override the URL defined in the WSDL
      - \c "headers": to override any HTTP headers sent in outgoing messages
      - \c "event_queue": to set an I/O event queue on the HTTPClient
      - \c "log": a @ref closure "closure" or @ref call_reference "call reference" taking a single string argument for
        informational logging
      - \c "dbglog": a @ref closure "closure" or @ref call_reference "call reference" taking a single string argument
        for verbose technical debugging logging

    Also the following keys can be set to set HTTP options:
    - \c "connect_timeout"
    - \c "http_version"
    - \c "max_redirects"
    - \c "proxy"
    - \c "timeout"

    @section soapclientrelnotes SoapClient Release Notes

    @subsection soapclient_1_0_1 SoapClient v1.0.1
    - allow connection options designating files to be selected as files
      (<a href="https://github.com/qorelanguage/qore/issues/4725">issue 4725</a>)

    @subsection soapclient_1_0 SoapClient v1.0
    - added information about data provider capabilities to connection info
      (<a href="https://github.com/qorelanguage/qore/issues/4576">issue 4576</a>)

    @subsection soapclient_0_5 SoapClient v0.5
    - implemented support for a data provider scheme cache and rich option information for connections
      (<a href="https://github.com/qorelanguage/qore/issues/4025">issue 4025</a>)

    @subsection soapclient_0_4 SoapClient v0.4
    - removed the obsolete \c SoapConnection::getConstructorInfo() method as connection serialization is a
      much more elegant and maintainable solution
      (<a href="https://github.com/qorelanguage/qore/issues/3696">issue 3696</a>)

    @subsection soapclient_0_3 SoapClient v0.3
    - added the @ref SoapClient::SoapConnection::getConstructorInfo() "SoapConnection::getConstructorInfo()"
      method to allow connections to be created dynamically, potentially in another process from a network
      call (<a href="https://github.com/qorelanguage/qore/issues/2628">issue 2628</a>)
    - fixed type error in exception handling in SOAP calls
      (<a href="https://github.com/qorelanguage/qore/issues/3336">issue 3336</a>)

    @subsection soapclient_0_2_5 SoapClient v0.2.5
    - added support for logging content of SOAP messages
    - added the @ref SoapClient::SoapConnection "SoapConnection" class

    @subsection soapclient_0_2_4_1 SoapClient v0.2.4.1
    - added the @ref SoapClient::SoapClient::callOperation() method
      (<a href="https://github.com/qorelanguage/qore/issues/1226">issue 1226</a>)
    - updated SOAP response processing to throw an exception when the server responds with an error code
      (<a href="https://github.com/qorelanguage/qore/issues/1228">issue 1228</a>)
    - fixed a typo in a debug logging statement
      (<a href="https://github.com/qorelanguage/qore/issues/1358">issue 1358</a>)
    - fixed and documented the "info" output hash format
      (<a href="https://github.com/qorelanguage/qore/issues/1359">issue 1359</a>)
    - fixed a bug in the SoapClient::constructor() where a WebService object was not supported
      (<a href="https://github.com/qorelanguage/qore/issues/1424">issue 1424</a>)

    @subsection soapclient_0_2_4 SoapClient v0.2.4
    - handle SOAP fault messages returned with 500-series error codes as Fault messages
    - added optional logging closures, added SOAP headers to info hash when available
    - fixed URI request path, fixed agent string
    - added the following methods:
      - @ref SoapClient::SoapClient::addDefaultHeaders()
      - @ref SoapClient::SoapClient::getDefaultHeaders()
      - @ref SoapClient::SoapClient::getSendEncoding()
      - @ref SoapClient::SoapClient::getWebService()
      - @ref SoapClient::SoapClient::setContentEncoding()
      - @ref SoapClient::SoapClient::setSendEncoding()

    @subsection soapclient_0_2_3 SoapClient v0.2.3
    - updated to a user module
*/

//! main SoapClient namespace
namespace SoapClient {
//! SOAP client class implementation, publically inherits qore's HTTPClient class public class SoapClient inherits HTTPClient {    //! version of the implementation of this class
    const Version = "1.0.1";

    //! default HTTP headers
    const Headers = {"User-Agent": ("Qore-Soap-Client/" + SoapClient::Version)};

    //! option keys passed to the HTTPClient constructor
    const HTTPOptions = keys HttpConnection::ConnectionScheme.options;

    //! @cond nodoc
    private {
        // web service definition
        WSDL::WebService wsdl;
        // service name
        string svc;
        string port;
        string binding;
        __7_ string url_path;

        __7_ code logc;
        __7_ code dbglogc;

        // send content encoding hash
        __7_ hash<auto> seh;
    }
    //! @endcond

    public {
        //! target URL
        string url;
        //! HTTP headers to use
        hash headers = Headers;

        //! Send content encoding options
        /** Send content encoding options are as follows:
            - \c "bzip": use bzip2 compression
            - \c "gzip": use gzip compression
            - \c "deflate": use deflate compression
            - \c "identity": use no content encoding
        */
        const EncodingSupport = ...;


        //! default threadhold for data compressions; transfers smaller than this size will not be compressed
        const CompressionThreshold = 1024;
    }

    //! creates the object based on a %WSDL which is parsed to a @ref WSDL::WebService "WebService" object which provides the basis for all communication with this object
    /** one of either the \c wsdl or \c wsdl_file keys is required in the hash given to the constructor or an exception will be thrown
        @param h valid option keys:
        - \c wsdl: the URL of the web service or a @ref WSDL::WebService "WebService" object itself
        - \c wsdl_file: a path to use to load the %WSDL and create the @ref WSDL::WebService "WebService" object
        - \c url: override the target URL given in the %WSDL
        - \c send_encoding: a @ref EncodingSupport "send data encoding option" or the value \c "auto" which means to use automatic encoding; if not present defaults to no content-encoding on sent message bodies
        - \c content_encoding: for possible values, see @ref EncodingSupport; this sets the send encoding (if the \c "send_encoding" option is not set) and the requested response encoding
        - [\c service]: in case multiple service entries are found in the WSDL, give the one to be used here
        - [\c port]: in case multiple port entries are found in the WSDL, give the one to be used here; note that the SOAP binding is resolved from the \a service and \a port options.
        - [\c log]: a log closure or call reference taking a single string giving the log message
        - [\c dbglog]: a log closure or call reference taking a single string giving the debug log message
        - also all options from @ref SoapClient::SoapClient::HTTPOptions "SoapClient::HTTPOptions", which are passed to @ref Qore::HTTPClient::constructor() "HTTPClient::constructor()"
    */
    constructor(hash<auto> h) ;


    //! returns a hash representing the serialized SOAP request for a given @ref WSDL::WSOperation "WSOperation"
    /** @param operation the SOAP operation to use to serialize the request; if the operation is not known to the underlying @ref WSDL::WebService "WebService" class and port/binding, an exception will be thrown
        @param args the arguments to the SOAP operation
        @param soap_header data structure for the SOAP header, if required by the message
        @param op a reference to return the @ref WSDL::WSOperation "WSOperation" object found
        @param http_header a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)
        @param xml_opts an integer XML generation option code; see @ref xml_generation_constants for possible values; combine multiple codes with binary or (\c |)
        @param soapaction an optional string that will override the SOAPAction for the request; en empty string here will prevent the SOAPAction from being sent

        @return a hash with the following keys:
        - \c hdr: a hash of message headers
        - \c body: the serialized message body
        - \c path: the path part of URL
        - \c method: the HTTP request method

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw WSDL-BINDING-ERROR the binding is not assigned to a SOAP operation in the WSDL

        @note content encoding is not applied here but rather internally by the call() methods
    */
    hash<auto> getMsg(string operation, auto args, __7_ hash<auto> soap_header, reference<auto> op, __7_ hash<auto> http_header, __7_ int xml_opts, __7_ string soapaction);


    //! makes a server call with the given operation, arguments, options, and optional info hash reference and returns the result
    /** @param operation the SOAP operation to use to serialize the request; if the operation is not known to the underlying @ref WSDL::WebService "WebService" class, an exception will be thrown
        @param args the arguments to the SOAP operation
        @param opts an optional hash of options for the call as follows:
        - \c soap_header: a hash giving SOAP header information, if required by the message
        - \c http_header: a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)
        - \c xml_opts: an integer XML generation option code; see @ref xml_generation_constants for possible values; combine multiple codes with binary or (\c |)
        - \c soapaction: an optional string that will override the SOAPAction for the request; en empty string here will prevent the SOAPAction from being sent
        - \c binding: the SOAP binding name, if not provided the first binding assigned to the operation will be used
        @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)

        @return a hash with the following keys:
        - \c hdr: a hash of message headers
        - \c body: the serialized message body

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw WSDL-BINDING-ERROR the binding is not assigned to a SOAP operation in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @note this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"

        @since %SoapClient 0.2.4.1
    */
    auto callOperation(string operation, auto args, __7_ hash<auto> opts, __7_ reference info);


    //! makes a server call with the given operation and arguments and returns the deserialized result
    /** @param operation the operation name for the SOAP call
        @param args the arguments to the SOAP operation
        @param soap_header optional soap headers (if required by the operation)
        @param http_header a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)
        @param binding the SOAP binding name, if not provided, the first binding assigned to the operation will be used

        @return the deserialized result of the SOAP call to the SOAP server

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw WSDL-BINDING-ERROR the binding is not assigned to a SOAP operation in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @see SoapClient::callOperation()

        @note
        - this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
        - it's recommended to use callOperation() instead, which was a cleaner and more extensible API
    */
    auto call(string operation, auto args, __7_ hash<auto> soap_header, __7_ hash<auto> http_header, __7_ string binding);


    //! makes a server call with the given operation and arguments and returns the deserialized result
    /** @param operation the operation name for the SOAP call
        @param args the arguments to the SOAP operation
        @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)
        @param binding the SOAP binding name, if not provided, the first binding assigned to the operation will be used

        @return the deserialized result of the SOAP call to the SOAP server

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw WSDL-BINDING-ERROR the binding is not assigned to a SOAP operation in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @see SoapClient::callOperation()

        @note
        - this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
        - it's recommended to use callOperation() instead, which was a cleaner and more extensible API
    */
    auto call(string operation, auto args, __7_ reference info, __7_ string binding);


    //! makes a server call with the given operation and arguments and returns the deserialized result with an output argument giving technical information about the call
    /** @param info an optional reference to return a hash of technical information about the SOAP call (raw message info and headers); the following keys are present in this hash:
        - \c "headers": a hash of HTTP request headers
        - \c "request-uri": the request URI string (ex: \c "POST /services/Soap/c/29.0 HTTP/1.1")
        - \c "response-uri": the response URI string (ex: \c "HTTP/1.1 200 OK")
        - \c "charset": the character encoding string (ex: \c "UTF-8")
        - \c "body-content-type": the \c Content-Type of the response without any \c charset declaration
        - \c "accept-charset": the valus of any \c Accept-Charset header in the response
        - \c "response-headers": a hash of HTTP response headers
        - \c "response-body": the raw XML response body (in case content encoding is used, this is the decoded value)
        - \c "request-body": the raw XML request body
        - \c "request-soap-headers": an optional hash of SOAP headers used in the request (if applicable)
        @param operation the operation name for the SOAP call
        @param args the arguments to the SOAP operation
        @param soap_header optional soap headers (if required by the operation)
        @param http_header a hash giving HTTP header information to include in the message (does not override automatically-generated SOAP message headers)

        @return the deserialized result of the SOAP call to the SOAP server

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw WSDL-BINDING-ERROR the binding is not assigned to a SOAP operation in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @see SoapClient::callOperation()

        @note
        - this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
        - it's recommended to use callOperation() instead, which was a cleaner and more extensible API
    */
    auto call(reference info, string operation, auto args, __7_ hash<auto> soap_header, __7_ hash<auto> http_header);


    //! makes the call to the SOAP server and ensures that SOAP fault responses returned with a 500-series status code are processed as a SOAP fault so that error information is returned in the resulting exception
private:
     auto makeCallIntern(__7_ reference info, string operation, auto args, __7_ hash<auto> opts);
public:


    //! uses SoapClient::call() to transparently serialize the argument and make a call to the given operation and return the deserialized results
    /** @param op the operation name, which is the method name passed to methodGate()
        @param ... zero or more arguments for the operation

        @return the deserialized result of the SOAP call to the SOAP server

        @throw WSDL-OPERATION-ERROR the operation is not defined in the WSDL
        @throw HTTP-CLIENT-RECEIVE-ERROR this exception is thrown when the SOAP server returns an HTTP error code; if a SOAP fault is returned, then it is deserialized and returned in the \a arg key of the exception hash

        @see SoapClient::callOperation()

        @note this method can throw any exception that @ref Qore::HTTPClient::send() "HTTPClient::send()" can throw as well as any XML parsing errors thrown by @ref Qore::XML::parse_xml() "parse_xml()"
    */
    auto methodGate(string op);


    //! returns a hash that can be used to ensure serialization with the XSD type given as the \a type argument
    hash<auto> getType(string type, auto v);


    //! returns the WSDL::WebService object associated with this object
    WSDL::WebService getWebService();


    //! returns a hash of information about the current WSDL
    /** @return a hash with the following keys:
        - \c "service": the name of the SOAP service used
        - \c "pore": the name of the SOAP port used
        - \c "binding": the name of the binding used
        - \c "url": the target URL
    */
    hash<auto> getInfo();


    //! change the data content encoding (compression) option for the object; see @ref EncodingSupport for valid options
    /** @par Example:
        @code{.py}
sc.setSendEncoding("gzip");
        @endcode

        The default is to send requests unencoded/uncompressed.

        @param enc the data content encoding (compression) option for the object; see @ref EncodingSupport for valid options; if the value \c "auto" is passed then \c "gzip" encoding is used

        @throw SOAPCLIENT-ERROR invalid or unsupported data content encoding / compression option

        @see
        - @ref setContentEncoding()
        - @ref getSendEncoding()
    */
    setSendEncoding(string enc = 'auto');


    //! sets the request and desired response encoding for the object; see @ref EncodingSupport for valid options
    /** @par Example:
        @code{.py}
soap.setContentEncoding("gzip");
        @endcode

        @param enc the data content encoding (compression) option for requests and the desired response content encoding for the object; see @ref EncodingSupport for valid options; if the value \c "auto" is passed then \c "gzip" encoding is used for outgoing requests and requested for responses

        @throw SOAPCLIENT-ERROR invalid or unsupported data content encoding / compression option

        @see
        - @ref getSendEncoding()
        - @ref setSendEncoding()

        @since %SoapClient 0.2.4
    */
    setContentEncoding(string enc = 'auto');


    //! adds default headers to each request; these headers will be sent in all requests but can be overridden in requests as well
    /** @par Example:
        @code{.py}
# disable gzip and bzip encoding in responses
soap.addDefaultHeaders(("Accept-Encoding": "compress"));
        @endcode

        @param h a hash of headers to add to the default headers to send on each request

        @note default headers can also be set in the constructor

        @see @ref SoapClient::SoapClient::getDefaultHeaders() "SoapClient::getDefaultHeaders()"

        @since %SoapClient 0.2.4
    */
    addDefaultHeaders(hash<auto> h);


    //! returns the hash of default headers to sent in all requests
    /** @par Example:
        @code{.py}
hash<auto> h = soap.getDefaultHeaders();
        @endcode

        @return the hash of default headers to sent in all requests

        @note default headers can be set in the constructor and in addDefaultHeaders()

        @see @ref getDefaultHeaders()

        @since %SoapClient 0.2.4
    */
    hash<auto> getDefaultHeaders();


    //! returns the current data content encoding (compression) object or @ref nothing if no encoding option is set; see @ref EncodingSupport for valid options
    /** @par Example:
        @code{.py}
*string ce = soap.getSendEncoding();
        @endcode

        @return the current data content encoding (compression) object or @ref nothing if no encoding option is set; see @ref EncodingSupport for valid options

        @see
        - @ref setContentEncoding()
        - @ref setSendEncoding()

        @since %SoapClient 0.2.4
    */
    __7_ string getSendEncoding();


    //! sends a log message to the log closure or call reference, if any
    log(string msg);


    //! sends a log message to the debug log closure or call reference, if any
    dbglog(string msg);

    /** Method is to be overriden to implement custom logging HTTP message sent and received from this object.
        @par msg information about message to be logged. There are keys 'reason' with value 'request', 'response' or 'error',
                'header' (HTTP headers, as received from the remote) and 'body' containing XML data in readable form, i.e. uncompressed, and, in case of
                a multipart message, only the related part is passed. Method is executed before message is sent or after
                has been received. Do not allow exception raising in method not to interrupt client function.

        @par Example:
        @code{.py}
private nothing msglog(hash<auto> msg) {
log(sprintf("%s:\n%s\n", msg.reason, msg.body));
}
        @endcode

        @since %SoapClient 0.2.5
    */
protected:
     nothing msglog(hash<auto> msg);
public:

}

//! class for SOAP connections; returns an object of class @ref SoapClient for connecting to SOAP servers
/** supports all @ref ConnectionProvider::HttpConnection "HttpConnection" options plus the following static
    initialization options:
    - \c "service": the name of the "service" to use (if more than 1 service is defined in the WSDL then this key is
      mandatory)
    - \c "port": in case multiple port entries are found in the WSDL, give the one to be used here
    - \c "target_url": overrides the URL in the WSDL (mapped to \c "url" in the
      @ref SoapClient::constructor(hash) "SoapClient::constructor()" argument)
    - \c "wsdl": overrides the WSDL URL provided by the connection URL; \c "wsdl" values with a \c "file://" scheme
      are subject to environment variable substitution

    @see @ref SoapClient::constructor(hash) for more information on the above options

    @note additionally supports the following runtime options in getImpl() for connection logging:
    - \c "log": a closure accepting a single string for logging
    - \c "dbglog": a closure taking a single string for detailed technical connection logging
*/
class SoapConnection : public HttpBasedConnection {

public:
        hash<auto> real_opts;

        //! Connection entry info
        const ConnectionScheme = ...;


    //! creates the SoapConnection object
    /** @param name the name of the connection
        @param description connection description
        @param url connection URL (potentially with password info)
        @param attributes various attributes. See below
        @param options connection options

        See @ref AbstractConnection::constructor() for \c attributes and \c options reference.
    */
    constructor(string name, string description, string url, hash<auto> attributes = {}, hash<auto> options = {})
 ;


    //! Creates the SoapConnection object
    /** @param config with the following keys:
        - name (required string): the connection name
        - display_name (optional string): the display name
        - short_desc (optional string): a short description in plain text
        - desc (optional string): a long description with markdown formatting
        - url (required string): the connection URL
        - opts (optional hash): connection options
        - logger (optional LoggerInterface object): logger for the connection
        @param attr optional connection attributes
        - monitor (optional bool): should the connection be monitored? Default: True
        - enabled (optional bool): is the connection enabled? Default: True
        - locked (optional bool): is the connection locked? Default: False
        - debug_data (optional bool): debug data? Default: False
        - tags (optional hash): tags for the connection; no default value

        @throw CONNECTION-OPTION-ERROR missing or invalid connection option or attribute
    */
    constructor(hash<auto> config, __7_ hash<auto> attr) ;


    //! returns \c "soap"
    string getType();


    //! returns runtime options
    /** @return a hash with the following keys reflecting support for the corresponding runtime options in getImpl() for connection logging:
        - \c "log": a closure accepting a single string for logging
        - \c "dbglog": a closure taking a single string for detailed technical connection logging
    */
    __7_ hash<auto> getRuntimeOptions();


    //! returns a data provider object for this connection
    /** @return a data provider object for this connection

        @see hasDataProvider()
    */
    DataProvider::AbstractDataProvider getDataProvider();


    //! returns @ref True as this connection returns a data provider with the @ref getDataProvider() method
    /** @return @ref True as this connection returns a data provider with the @ref getDataProvider() method

        @see @ref getDataProvider()
    */
    bool hasDataProvider();


    //! returns @ref False
    /** This connection type has not yet been updated to support the %Qore @ref Qore::Socket "Socket-based" polling
        API

        @return @ref False
    */
    bool supportsPollingApi();


    //! Common constructor initialization
protected:
     soapSetup();
public:


    //! Set child capabilities
protected:
     setChildCapabilities();
public:


    //! returns a @ref SoapClient object
    /** @param connect if @ref Qore::True "True", then @ref Qore::HTTPClient::connect() "SoapClient::connect()" is called
        @param rtopts supports the following runtime options in getImpl() for connection logging:
        - \c "log": a closure accepting a single string for logging
        - \c "dbglog": a closure taking a single string for detailed technical connection logging

        @return a @ref SoapClient object
    */
protected:
     SoapClient getImpl(bool connect = True, __7_ hash<auto> rtopts);
public:


    //! Returns the ConnectionSchemeInfo hash for this object
protected:
     hash<ConnectionSchemeInfo> getConnectionSchemeInfoImpl();
public:

};
}
