/* Copyright (c) 2014, Google Inc.
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. */

#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <utility>
#include <vector>

#include <gtest/gtest.h>

#include <openssl/bytestring.h>
#include <openssl/crypto.h>
#include <openssl/digest.h>
#include <openssl/err.h>
#include <openssl/pkcs8.h>
#include <openssl/rsa.h>

#include "../internal.h"
#include "../test/test_util.h"


// kExampleRSAKeyDER is an RSA private key in ASN.1, DER format. Of course, you
// should never use this key anywhere but in an example.
static const uint8_t kExampleRSAKeyDER[] = {
    0x30, 0x82, 0x02, 0x5c, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81, 0x00, 0xf8,
    0xb8, 0x6c, 0x83, 0xb4, 0xbc, 0xd9, 0xa8, 0x57, 0xc0, 0xa5, 0xb4, 0x59,
    0x76, 0x8c, 0x54, 0x1d, 0x79, 0xeb, 0x22, 0x52, 0x04, 0x7e, 0xd3, 0x37,
    0xeb, 0x41, 0xfd, 0x83, 0xf9, 0xf0, 0xa6, 0x85, 0x15, 0x34, 0x75, 0x71,
    0x5a, 0x84, 0xa8, 0x3c, 0xd2, 0xef, 0x5a, 0x4e, 0xd3, 0xde, 0x97, 0x8a,
    0xdd, 0xff, 0xbb, 0xcf, 0x0a, 0xaa, 0x86, 0x92, 0xbe, 0xb8, 0x50, 0xe4,
    0xcd, 0x6f, 0x80, 0x33, 0x30, 0x76, 0x13, 0x8f, 0xca, 0x7b, 0xdc, 0xec,
    0x5a, 0xca, 0x63, 0xc7, 0x03, 0x25, 0xef, 0xa8, 0x8a, 0x83, 0x58, 0x76,
    0x20, 0xfa, 0x16, 0x77, 0xd7, 0x79, 0x92, 0x63, 0x01, 0x48, 0x1a, 0xd8,
    0x7b, 0x67, 0xf1, 0x52, 0x55, 0x49, 0x4e, 0xd6, 0x6e, 0x4a, 0x5c, 0xd7,
    0x7a, 0x37, 0x36, 0x0c, 0xde, 0xdd, 0x8f, 0x44, 0xe8, 0xc2, 0xa7, 0x2c,
    0x2b, 0xb5, 0xaf, 0x64, 0x4b, 0x61, 0x07, 0x02, 0x03, 0x01, 0x00, 0x01,
    0x02, 0x81, 0x80, 0x74, 0x88, 0x64, 0x3f, 0x69, 0x45, 0x3a, 0x6d, 0xc7,
    0x7f, 0xb9, 0xa3, 0xc0, 0x6e, 0xec, 0xdc, 0xd4, 0x5a, 0xb5, 0x32, 0x85,
    0x5f, 0x19, 0xd4, 0xf8, 0xd4, 0x3f, 0x3c, 0xfa, 0xc2, 0xf6, 0x5f, 0xee,
    0xe6, 0xba, 0x87, 0x74, 0x2e, 0xc7, 0x0c, 0xd4, 0x42, 0xb8, 0x66, 0x85,
    0x9c, 0x7b, 0x24, 0x61, 0xaa, 0x16, 0x11, 0xf6, 0xb5, 0xb6, 0xa4, 0x0a,
    0xc9, 0x55, 0x2e, 0x81, 0xa5, 0x47, 0x61, 0xcb, 0x25, 0x8f, 0xc2, 0x15,
    0x7b, 0x0e, 0x7c, 0x36, 0x9f, 0x3a, 0xda, 0x58, 0x86, 0x1c, 0x5b, 0x83,
    0x79, 0xe6, 0x2b, 0xcc, 0xe6, 0xfa, 0x2c, 0x61, 0xf2, 0x78, 0x80, 0x1b,
    0xe2, 0xf3, 0x9d, 0x39, 0x2b, 0x65, 0x57, 0x91, 0x3d, 0x71, 0x99, 0x73,
    0xa5, 0xc2, 0x79, 0x20, 0x8c, 0x07, 0x4f, 0xe5, 0xb4, 0x60, 0x1f, 0x99,
    0xa2, 0xb1, 0x4f, 0x0c, 0xef, 0xbc, 0x59, 0x53, 0x00, 0x7d, 0xb1, 0x02,
    0x41, 0x00, 0xfc, 0x7e, 0x23, 0x65, 0x70, 0xf8, 0xce, 0xd3, 0x40, 0x41,
    0x80, 0x6a, 0x1d, 0x01, 0xd6, 0x01, 0xff, 0xb6, 0x1b, 0x3d, 0x3d, 0x59,
    0x09, 0x33, 0x79, 0xc0, 0x4f, 0xde, 0x96, 0x27, 0x4b, 0x18, 0xc6, 0xd9,
    0x78, 0xf1, 0xf4, 0x35, 0x46, 0xe9, 0x7c, 0x42, 0x7a, 0x5d, 0x9f, 0xef,
    0x54, 0xb8, 0xf7, 0x9f, 0xc4, 0x33, 0x6c, 0xf3, 0x8c, 0x32, 0x46, 0x87,
    0x67, 0x30, 0x7b, 0xa7, 0xac, 0xe3, 0x02, 0x41, 0x00, 0xfc, 0x2c, 0xdf,
    0x0c, 0x0d, 0x88, 0xf5, 0xb1, 0x92, 0xa8, 0x93, 0x47, 0x63, 0x55, 0xf5,
    0xca, 0x58, 0x43, 0xba, 0x1c, 0xe5, 0x9e, 0xb6, 0x95, 0x05, 0xcd, 0xb5,
    0x82, 0xdf, 0xeb, 0x04, 0x53, 0x9d, 0xbd, 0xc2, 0x38, 0x16, 0xb3, 0x62,
    0xdd, 0xa1, 0x46, 0xdb, 0x6d, 0x97, 0x93, 0x9f, 0x8a, 0xc3, 0x9b, 0x64,
    0x7e, 0x42, 0xe3, 0x32, 0x57, 0x19, 0x1b, 0xd5, 0x6e, 0x85, 0xfa, 0xb8,
    0x8d, 0x02, 0x41, 0x00, 0xbc, 0x3d, 0xde, 0x6d, 0xd6, 0x97, 0xe8, 0xba,
    0x9e, 0x81, 0x37, 0x17, 0xe5, 0xa0, 0x64, 0xc9, 0x00, 0xb7, 0xe7, 0xfe,
    0xf4, 0x29, 0xd9, 0x2e, 0x43, 0x6b, 0x19, 0x20, 0xbd, 0x99, 0x75, 0xe7,
    0x76, 0xf8, 0xd3, 0xae, 0xaf, 0x7e, 0xb8, 0xeb, 0x81, 0xf4, 0x9d, 0xfe,
    0x07, 0x2b, 0x0b, 0x63, 0x0b, 0x5a, 0x55, 0x90, 0x71, 0x7d, 0xf1, 0xdb,
    0xd9, 0xb1, 0x41, 0x41, 0x68, 0x2f, 0x4e, 0x39, 0x02, 0x40, 0x5a, 0x34,
    0x66, 0xd8, 0xf5, 0xe2, 0x7f, 0x18, 0xb5, 0x00, 0x6e, 0x26, 0x84, 0x27,
    0x14, 0x93, 0xfb, 0xfc, 0xc6, 0x0f, 0x5e, 0x27, 0xe6, 0xe1, 0xe9, 0xc0,
    0x8a, 0xe4, 0x34, 0xda, 0xe9, 0xa2, 0x4b, 0x73, 0xbc, 0x8c, 0xb9, 0xba,
    0x13, 0x6c, 0x7a, 0x2b, 0x51, 0x84, 0xa3, 0x4a, 0xe0, 0x30, 0x10, 0x06,
    0x7e, 0xed, 0x17, 0x5a, 0x14, 0x00, 0xc9, 0xef, 0x85, 0xea, 0x52, 0x2c,
    0xbc, 0x65, 0x02, 0x40, 0x51, 0xe3, 0xf2, 0x83, 0x19, 0x9b, 0xc4, 0x1e,
    0x2f, 0x50, 0x3d, 0xdf, 0x5a, 0xa2, 0x18, 0xca, 0x5f, 0x2e, 0x49, 0xaf,
    0x6f, 0xcc, 0xfa, 0x65, 0x77, 0x94, 0xb5, 0xa1, 0x0a, 0xa9, 0xd1, 0x8a,
    0x39, 0x37, 0xf4, 0x0b, 0xa0, 0xd7, 0x82, 0x27, 0x5e, 0xae, 0x17, 0x17,
    0xa1, 0x1e, 0x54, 0x34, 0xbf, 0x6e, 0xc4, 0x8e, 0x99, 0x5d, 0x08, 0xf1,
    0x2d, 0x86, 0x9d, 0xa5, 0x20, 0x1b, 0xe5, 0xdf,
};

static const uint8_t kExampleDSAKeyDER[] = {
    0x30, 0x82, 0x03, 0x56, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00,
    0x9e, 0x12, 0xfa, 0xb3, 0xde, 0x12, 0x21, 0x35, 0x01, 0xdd, 0x82, 0xaa,
    0x10, 0xca, 0x2d, 0x10, 0x1d, 0x2d, 0x4e, 0xbf, 0xef, 0x4d, 0x2a, 0x3f,
    0x8d, 0xaa, 0x0f, 0xe0, 0xce, 0xda, 0xd8, 0xd6, 0xaf, 0x85, 0x61, 0x6a,
    0xa2, 0xf3, 0x25, 0x2c, 0x0a, 0x2b, 0x5a, 0x6d, 0xb0, 0x9e, 0x6f, 0x14,
    0x90, 0x0e, 0x0d, 0xdb, 0x83, 0x11, 0x87, 0x6d, 0xd8, 0xf9, 0x66, 0x95,
    0x25, 0xf9, 0x9e, 0xd6, 0x59, 0x49, 0xe1, 0x84, 0xd5, 0x06, 0x47, 0x93,
    0x27, 0x11, 0x69, 0xa2, 0x28, 0x68, 0x0b, 0x95, 0xec, 0x12, 0xf5, 0x9a,
    0x8e, 0x20, 0xb2, 0x1f, 0x2b, 0x58, 0xeb, 0x2a, 0x20, 0x12, 0xd3, 0x5b,
    0xde, 0x2e, 0xe3, 0x51, 0x82, 0x2f, 0xe8, 0xf3, 0x2d, 0x0a, 0x33, 0x05,
    0x65, 0xdc, 0xce, 0x5c, 0x67, 0x2b, 0x72, 0x59, 0xc1, 0x4b, 0x24, 0x33,
    0xd0, 0xb5, 0xb2, 0xca, 0x2b, 0x2d, 0xb0, 0xab, 0x62, 0x6e, 0x8f, 0x13,
    0xf4, 0x7f, 0xe0, 0x34, 0x5d, 0x90, 0x4e, 0x72, 0x94, 0xbb, 0x03, 0x8e,
    0x9c, 0xe2, 0x1a, 0x9e, 0x58, 0x0b, 0x83, 0x35, 0x62, 0x78, 0x70, 0x6c,
    0xfe, 0x76, 0x84, 0x36, 0xc6, 0x9d, 0xe1, 0x49, 0xcc, 0xff, 0x98, 0xb4,
    0xaa, 0xb8, 0xcb, 0x4f, 0x63, 0x85, 0xc9, 0xf1, 0x02, 0xce, 0x59, 0x34,
    0x6e, 0xae, 0xef, 0x27, 0xe0, 0xad, 0x22, 0x2d, 0x53, 0xd6, 0xe8, 0x9c,
    0xc8, 0xcd, 0xe5, 0x77, 0x6d, 0xd0, 0x00, 0x57, 0xb0, 0x3f, 0x2d, 0x88,
    0xab, 0x3c, 0xed, 0xba, 0xfd, 0x7b, 0x58, 0x5f, 0x0b, 0x7f, 0x78, 0x35,
    0xe1, 0x7a, 0x37, 0x28, 0xbb, 0xf2, 0x5e, 0xa6, 0x25, 0x72, 0xf2, 0x45,
    0xdc, 0x11, 0x1f, 0x3c, 0xe3, 0x9c, 0xb6, 0xff, 0xac, 0xc3, 0x1b, 0x0a,
    0x27, 0x90, 0xe7, 0xbd, 0xe9, 0x02, 0x24, 0xea, 0x9b, 0x09, 0x31, 0x53,
    0x62, 0xaf, 0x3d, 0x2b, 0x02, 0x21, 0x00, 0xf3, 0x81, 0xdc, 0xf5, 0x3e,
    0xbf, 0x72, 0x4f, 0x8b, 0x2e, 0x5c, 0xa8, 0x2c, 0x01, 0x0f, 0xb4, 0xb5,
    0xed, 0xa9, 0x35, 0x8d, 0x0f, 0xd8, 0x8e, 0xd2, 0x78, 0x58, 0x94, 0x88,
    0xb5, 0x4f, 0xc3, 0x02, 0x82, 0x01, 0x00, 0x0c, 0x40, 0x2a, 0x72, 0x5d,
    0xcc, 0x3a, 0x62, 0xe0, 0x2b, 0xf4, 0xcf, 0x43, 0xcd, 0x17, 0xf4, 0xa4,
    0x93, 0x59, 0x12, 0x20, 0x22, 0x36, 0x69, 0xcf, 0x41, 0x93, 0xed, 0xab,
    0x42, 0x3a, 0xd0, 0x8d, 0xfb, 0x55, 0x2e, 0x30, 0x8a, 0x6a, 0x57, 0xa5,
    0xff, 0xbc, 0x7c, 0xd0, 0xfb, 0x20, 0x87, 0xf8, 0x1f, 0x8d, 0xf0, 0xcb,
    0x08, 0xab, 0x21, 0x33, 0x28, 0x7d, 0x2b, 0x69, 0x68, 0x71, 0x4a, 0x94,
    0xf6, 0x33, 0xc9, 0x40, 0x84, 0x5a, 0x48, 0xa3, 0xe1, 0x67, 0x08, 0xdd,
    0xe7, 0x61, 0xcc, 0x6a, 0x8e, 0xab, 0x2d, 0x84, 0xdb, 0x21, 0xb6, 0xea,
    0x5b, 0x07, 0x68, 0x14, 0x93, 0xcc, 0x9c, 0x31, 0xfb, 0xc3, 0x68, 0xb2,
    0x43, 0xf6, 0xdd, 0xf8, 0xc9, 0x32, 0xa8, 0xb4, 0x03, 0x8f, 0x44, 0xe7,
    0xb1, 0x5c, 0xa8, 0x76, 0x34, 0x4a, 0x14, 0x78, 0x59, 0xf2, 0xb4, 0x3b,
    0x39, 0x45, 0x86, 0x68, 0xad, 0x5e, 0x0a, 0x1a, 0x9a, 0x66, 0x95, 0x46,
    0xdd, 0x28, 0x12, 0xe3, 0xb3, 0x61, 0x7a, 0x0a, 0xef, 0x99, 0xd5, 0x8e,
    0x3b, 0xb4, 0xcc, 0x87, 0xfd, 0x94, 0x22, 0x5e, 0x01, 0xd2, 0xdc, 0xc4,
    0x69, 0xa7, 0x72, 0x68, 0x14, 0x6c, 0x51, 0x91, 0x8f, 0x18, 0xe8, 0xb4,
    0xd7, 0x0a, 0xa1, 0xf0, 0xc7, 0x62, 0x3b, 0xcc, 0x52, 0xcf, 0x37, 0x31,
    0xd3, 0x86, 0x41, 0xb2, 0xd2, 0x83, 0x0b, 0x7e, 0xec, 0xb2, 0xf0, 0x95,
    0x52, 0xff, 0x13, 0x7d, 0x04, 0x6e, 0x49, 0x4e, 0x7f, 0x33, 0xc3, 0x59,
    0x00, 0x02, 0xb1, 0x6d, 0x1b, 0x97, 0xd9, 0x36, 0xfd, 0xa2, 0x8f, 0x90,
    0xc3, 0xed, 0x3c, 0xa3, 0x53, 0x38, 0x16, 0x8a, 0xc1, 0x6f, 0x77, 0xc3,
    0xc5, 0x7a, 0xdc, 0x2e, 0x8f, 0x7c, 0x6c, 0x22, 0x56, 0xe4, 0x1a, 0x5f,
    0x65, 0x45, 0x05, 0x90, 0xdb, 0xb5, 0xbc, 0xf0, 0x6d, 0x66, 0x61, 0x02,
    0x82, 0x01, 0x00, 0x31, 0x97, 0x31, 0xa1, 0x4e, 0x38, 0x56, 0x88, 0xdb,
    0x94, 0x1d, 0xbf, 0x65, 0x5c, 0xda, 0x4b, 0xc2, 0x10, 0xde, 0x74, 0x20,
    0x03, 0xce, 0x13, 0x60, 0xf2, 0x25, 0x1d, 0x55, 0x7c, 0x5d, 0x94, 0x82,
    0x54, 0x08, 0x53, 0xdb, 0x85, 0x95, 0xbf, 0xdd, 0x5e, 0x50, 0xd5, 0x96,
    0xe0, 0x79, 0x51, 0x1b, 0xbf, 0x4d, 0x4e, 0xb9, 0x3a, 0xc5, 0xee, 0xc4,
    0x5e, 0x98, 0x75, 0x7b, 0xbe, 0xff, 0x30, 0xe6, 0xd0, 0x7b, 0xa6, 0xf1,
    0xbc, 0x29, 0xea, 0xdf, 0xec, 0xf3, 0x8b, 0xfa, 0x83, 0x11, 0x9f, 0x3f,
    0xf0, 0x5d, 0x06, 0x51, 0x32, 0xaa, 0x21, 0xfc, 0x26, 0x17, 0xe7, 0x50,
    0xc2, 0x16, 0xba, 0xfa, 0x54, 0xb7, 0x7e, 0x1d, 0x2c, 0xa6, 0xa3, 0x41,
    0x66, 0x33, 0x94, 0x83, 0xb9, 0xbf, 0xa0, 0x4f, 0xbd, 0xa6, 0xfd, 0x2c,
    0x81, 0x58, 0x35, 0x33, 0x39, 0xc0, 0x6d, 0x33, 0x40, 0x56, 0x64, 0x12,
    0x5a, 0xcd, 0x35, 0x53, 0x21, 0x78, 0x8f, 0x27, 0x24, 0x37, 0x66, 0x8a,
    0xdf, 0x5e, 0x5f, 0x63, 0xfc, 0x8b, 0x2d, 0xef, 0x57, 0xdb, 0x40, 0x25,
    0xd5, 0x17, 0x53, 0x0b, 0xe4, 0xa5, 0xae, 0x54, 0xbf, 0x46, 0x4f, 0xa6,
    0x79, 0xc3, 0x74, 0xfa, 0x1f, 0x85, 0x34, 0x64, 0x6d, 0xc5, 0x03, 0xeb,
    0x72, 0x98, 0x80, 0x7b, 0xc0, 0x8f, 0x35, 0x11, 0xa7, 0x09, 0xeb, 0x51,
    0xe0, 0xb0, 0xac, 0x92, 0x14, 0xf2, 0xad, 0x37, 0x95, 0x5a, 0xba, 0x8c,
    0xc4, 0xdb, 0xed, 0xc4, 0x4e, 0x8b, 0x8f, 0x84, 0x33, 0x64, 0xf8, 0x57,
    0x12, 0xd7, 0x08, 0x7e, 0x90, 0x66, 0xdf, 0x91, 0x50, 0x23, 0xf2, 0x73,
    0xc0, 0x6b, 0xb1, 0x15, 0xdd, 0x64, 0xd7, 0xc9, 0x75, 0x17, 0x73, 0x72,
    0xda, 0x33, 0xc4, 0x6f, 0xa5, 0x47, 0xa1, 0xcc, 0xd1, 0xc6, 0x62, 0xe5,
    0xca, 0xab, 0x5f, 0x2a, 0x8f, 0x6b, 0xcc, 0x02, 0x21, 0x00, 0xb0, 0xc7,
    0x68, 0x70, 0x27, 0x43, 0xbc, 0x51, 0x24, 0x29, 0x93, 0xa9, 0x71, 0xa5,
    0x28, 0x89, 0x79, 0x54, 0x44, 0xf7, 0xc6, 0x45, 0x22, 0x03, 0xd0, 0xce,
    0x84, 0xfe, 0x61, 0x17, 0xd4, 0x6e,
};

static const uint8_t kMsg[] = {1, 2, 3, 4};

static const uint8_t kSignature[] = {
    0xa5, 0xf0, 0x8a, 0x47, 0x5d, 0x3c, 0xb3, 0xcc, 0xa9, 0x79, 0xaf, 0x4d,
    0x8c, 0xae, 0x4c, 0x14, 0xef, 0xc2, 0x0b, 0x34, 0x36, 0xde, 0xf4, 0x3e,
    0x3d, 0xbb, 0x4a, 0x60, 0x5c, 0xc8, 0x91, 0x28, 0xda, 0xfb, 0x7e, 0x04,
    0x96, 0x7e, 0x63, 0x13, 0x90, 0xce, 0xb9, 0xb4, 0x62, 0x7a, 0xfd, 0x09,
    0x3d, 0xc7, 0x67, 0x78, 0x54, 0x04, 0xeb, 0x52, 0x62, 0x6e, 0x24, 0x67,
    0xb4, 0x40, 0xfc, 0x57, 0x62, 0xc6, 0xf1, 0x67, 0xc1, 0x97, 0x8f, 0x6a,
    0xa8, 0xae, 0x44, 0x46, 0x5e, 0xab, 0x67, 0x17, 0x53, 0x19, 0x3a, 0xda,
    0x5a, 0xc8, 0x16, 0x3e, 0x86, 0xd5, 0xc5, 0x71, 0x2f, 0xfc, 0x23, 0x48,
    0xd9, 0x0b, 0x13, 0xdd, 0x7b, 0x5a, 0x25, 0x79, 0xef, 0xa5, 0x7b, 0x04,
    0xed, 0x44, 0xf6, 0x18, 0x55, 0xe4, 0x0a, 0xe9, 0x57, 0x79, 0x5d, 0xd7,
    0x55, 0xa7, 0xab, 0x45, 0x02, 0x97, 0x60, 0x42,
};

// kExampleRSAKeyPKCS8 is kExampleRSAKeyDER encoded in a PKCS #8
// PrivateKeyInfo.
static const uint8_t kExampleRSAKeyPKCS8[] = {
    0x30, 0x82, 0x02, 0x76, 0x02, 0x01, 0x00, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x02, 0x60, 0x30, 0x82, 0x02, 0x5c, 0x02, 0x01, 0x00, 0x02, 0x81, 0x81,
    0x00, 0xf8, 0xb8, 0x6c, 0x83, 0xb4, 0xbc, 0xd9, 0xa8, 0x57, 0xc0, 0xa5,
    0xb4, 0x59, 0x76, 0x8c, 0x54, 0x1d, 0x79, 0xeb, 0x22, 0x52, 0x04, 0x7e,
    0xd3, 0x37, 0xeb, 0x41, 0xfd, 0x83, 0xf9, 0xf0, 0xa6, 0x85, 0x15, 0x34,
    0x75, 0x71, 0x5a, 0x84, 0xa8, 0x3c, 0xd2, 0xef, 0x5a, 0x4e, 0xd3, 0xde,
    0x97, 0x8a, 0xdd, 0xff, 0xbb, 0xcf, 0x0a, 0xaa, 0x86, 0x92, 0xbe, 0xb8,
    0x50, 0xe4, 0xcd, 0x6f, 0x80, 0x33, 0x30, 0x76, 0x13, 0x8f, 0xca, 0x7b,
    0xdc, 0xec, 0x5a, 0xca, 0x63, 0xc7, 0x03, 0x25, 0xef, 0xa8, 0x8a, 0x83,
    0x58, 0x76, 0x20, 0xfa, 0x16, 0x77, 0xd7, 0x79, 0x92, 0x63, 0x01, 0x48,
    0x1a, 0xd8, 0x7b, 0x67, 0xf1, 0x52, 0x55, 0x49, 0x4e, 0xd6, 0x6e, 0x4a,
    0x5c, 0xd7, 0x7a, 0x37, 0x36, 0x0c, 0xde, 0xdd, 0x8f, 0x44, 0xe8, 0xc2,
    0xa7, 0x2c, 0x2b, 0xb5, 0xaf, 0x64, 0x4b, 0x61, 0x07, 0x02, 0x03, 0x01,
    0x00, 0x01, 0x02, 0x81, 0x80, 0x74, 0x88, 0x64, 0x3f, 0x69, 0x45, 0x3a,
    0x6d, 0xc7, 0x7f, 0xb9, 0xa3, 0xc0, 0x6e, 0xec, 0xdc, 0xd4, 0x5a, 0xb5,
    0x32, 0x85, 0x5f, 0x19, 0xd4, 0xf8, 0xd4, 0x3f, 0x3c, 0xfa, 0xc2, 0xf6,
    0x5f, 0xee, 0xe6, 0xba, 0x87, 0x74, 0x2e, 0xc7, 0x0c, 0xd4, 0x42, 0xb8,
    0x66, 0x85, 0x9c, 0x7b, 0x24, 0x61, 0xaa, 0x16, 0x11, 0xf6, 0xb5, 0xb6,
    0xa4, 0x0a, 0xc9, 0x55, 0x2e, 0x81, 0xa5, 0x47, 0x61, 0xcb, 0x25, 0x8f,
    0xc2, 0x15, 0x7b, 0x0e, 0x7c, 0x36, 0x9f, 0x3a, 0xda, 0x58, 0x86, 0x1c,
    0x5b, 0x83, 0x79, 0xe6, 0x2b, 0xcc, 0xe6, 0xfa, 0x2c, 0x61, 0xf2, 0x78,
    0x80, 0x1b, 0xe2, 0xf3, 0x9d, 0x39, 0x2b, 0x65, 0x57, 0x91, 0x3d, 0x71,
    0x99, 0x73, 0xa5, 0xc2, 0x79, 0x20, 0x8c, 0x07, 0x4f, 0xe5, 0xb4, 0x60,
    0x1f, 0x99, 0xa2, 0xb1, 0x4f, 0x0c, 0xef, 0xbc, 0x59, 0x53, 0x00, 0x7d,
    0xb1, 0x02, 0x41, 0x00, 0xfc, 0x7e, 0x23, 0x65, 0x70, 0xf8, 0xce, 0xd3,
    0x40, 0x41, 0x80, 0x6a, 0x1d, 0x01, 0xd6, 0x01, 0xff, 0xb6, 0x1b, 0x3d,
    0x3d, 0x59, 0x09, 0x33, 0x79, 0xc0, 0x4f, 0xde, 0x96, 0x27, 0x4b, 0x18,
    0xc6, 0xd9, 0x78, 0xf1, 0xf4, 0x35, 0x46, 0xe9, 0x7c, 0x42, 0x7a, 0x5d,
    0x9f, 0xef, 0x54, 0xb8, 0xf7, 0x9f, 0xc4, 0x33, 0x6c, 0xf3, 0x8c, 0x32,
    0x46, 0x87, 0x67, 0x30, 0x7b, 0xa7, 0xac, 0xe3, 0x02, 0x41, 0x00, 0xfc,
    0x2c, 0xdf, 0x0c, 0x0d, 0x88, 0xf5, 0xb1, 0x92, 0xa8, 0x93, 0x47, 0x63,
    0x55, 0xf5, 0xca, 0x58, 0x43, 0xba, 0x1c, 0xe5, 0x9e, 0xb6, 0x95, 0x05,
    0xcd, 0xb5, 0x82, 0xdf, 0xeb, 0x04, 0x53, 0x9d, 0xbd, 0xc2, 0x38, 0x16,
    0xb3, 0x62, 0xdd, 0xa1, 0x46, 0xdb, 0x6d, 0x97, 0x93, 0x9f, 0x8a, 0xc3,
    0x9b, 0x64, 0x7e, 0x42, 0xe3, 0x32, 0x57, 0x19, 0x1b, 0xd5, 0x6e, 0x85,
    0xfa, 0xb8, 0x8d, 0x02, 0x41, 0x00, 0xbc, 0x3d, 0xde, 0x6d, 0xd6, 0x97,
    0xe8, 0xba, 0x9e, 0x81, 0x37, 0x17, 0xe5, 0xa0, 0x64, 0xc9, 0x00, 0xb7,
    0xe7, 0xfe, 0xf4, 0x29, 0xd9, 0x2e, 0x43, 0x6b, 0x19, 0x20, 0xbd, 0x99,
    0x75, 0xe7, 0x76, 0xf8, 0xd3, 0xae, 0xaf, 0x7e, 0xb8, 0xeb, 0x81, 0xf4,
    0x9d, 0xfe, 0x07, 0x2b, 0x0b, 0x63, 0x0b, 0x5a, 0x55, 0x90, 0x71, 0x7d,
    0xf1, 0xdb, 0xd9, 0xb1, 0x41, 0x41, 0x68, 0x2f, 0x4e, 0x39, 0x02, 0x40,
    0x5a, 0x34, 0x66, 0xd8, 0xf5, 0xe2, 0x7f, 0x18, 0xb5, 0x00, 0x6e, 0x26,
    0x84, 0x27, 0x14, 0x93, 0xfb, 0xfc, 0xc6, 0x0f, 0x5e, 0x27, 0xe6, 0xe1,
    0xe9, 0xc0, 0x8a, 0xe4, 0x34, 0xda, 0xe9, 0xa2, 0x4b, 0x73, 0xbc, 0x8c,
    0xb9, 0xba, 0x13, 0x6c, 0x7a, 0x2b, 0x51, 0x84, 0xa3, 0x4a, 0xe0, 0x30,
    0x10, 0x06, 0x7e, 0xed, 0x17, 0x5a, 0x14, 0x00, 0xc9, 0xef, 0x85, 0xea,
    0x52, 0x2c, 0xbc, 0x65, 0x02, 0x40, 0x51, 0xe3, 0xf2, 0x83, 0x19, 0x9b,
    0xc4, 0x1e, 0x2f, 0x50, 0x3d, 0xdf, 0x5a, 0xa2, 0x18, 0xca, 0x5f, 0x2e,
    0x49, 0xaf, 0x6f, 0xcc, 0xfa, 0x65, 0x77, 0x94, 0xb5, 0xa1, 0x0a, 0xa9,
    0xd1, 0x8a, 0x39, 0x37, 0xf4, 0x0b, 0xa0, 0xd7, 0x82, 0x27, 0x5e, 0xae,
    0x17, 0x17, 0xa1, 0x1e, 0x54, 0x34, 0xbf, 0x6e, 0xc4, 0x8e, 0x99, 0x5d,
    0x08, 0xf1, 0x2d, 0x86, 0x9d, 0xa5, 0x20, 0x1b, 0xe5, 0xdf,
};

// kExampleRSAPSSKeyNoPSSParams is a sample RSAPSS private key.
// This private key has rsaPSS oid but no pss parameters.
// This key is copied from https://github.com/aws/s2n-tls/blame/c4d90e34fbd2ba64bb17a95628622ccc1d0c6807/tests/pems/rsa_pss_2048_sha256_CA_key.pem#L1
static const uint8_t kExampleRSAPSSKeyNoPSSParams[] = {
    0x30, 0x82, 0x04, 0xbb, 0x02, 0x01, 0x00, 0x30, 0x0b, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x04, 0x82, 0x04, 0xa7,
    0x30, 0x82, 0x04, 0xa3, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00,
    0xde, 0x6b, 0x53, 0x5c, 0x76, 0xb4, 0xc5, 0xc3, 0xf4, 0x22, 0x5d, 0xe4,
    0xa4, 0x6c, 0x7f, 0xe1, 0x4b, 0x0c, 0x7d, 0x28, 0xcf, 0xe5, 0x4f, 0xc3,
    0x48, 0x84, 0xb3, 0x90, 0x68, 0x23, 0xe4, 0x3a, 0xdf, 0xf5, 0x5b, 0x6e,
    0xae, 0xf9, 0x04, 0xe7, 0x3f, 0x8f, 0xbf, 0x07, 0xad, 0x90, 0x75, 0xa0,
    0x35, 0x17, 0x17, 0x6d, 0xb1, 0x7a, 0x56, 0x63, 0xf2, 0xcc, 0xc4, 0x26,
    0x23, 0xcd, 0x67, 0x82, 0x8d, 0x1c, 0x88, 0x51, 0x85, 0xfe, 0x58, 0x08,
    0x2d, 0xc4, 0x4e, 0x45, 0x22, 0x1e, 0xd8, 0x53, 0x79, 0xc7, 0x4c, 0xbe,
    0x24, 0x14, 0xed, 0x82, 0xf2, 0x96, 0xf9, 0x59, 0x71, 0x51, 0x94, 0x15,
    0xbd, 0xe2, 0x67, 0x26, 0x27, 0xda, 0x8c, 0x33, 0x2a, 0xd6, 0x19, 0xfa,
    0xf3, 0x83, 0xb8, 0x8f, 0x6e, 0x1d, 0xda, 0xd8, 0xa1, 0xcc, 0x33, 0xbb,
    0x90, 0x6b, 0xc8, 0xac, 0x12, 0x27, 0x4e, 0xd5, 0x11, 0x32, 0x6f, 0x2b,
    0xf3, 0x52, 0x07, 0x0e, 0xf5, 0x2c, 0x97, 0xb8, 0x7b, 0x17, 0xc6, 0x18,
    0x94, 0xad, 0x2e, 0x52, 0x14, 0x6c, 0x09, 0x0d, 0xfc, 0x05, 0xd2, 0x9d,
    0xf7, 0xec, 0xa5, 0x44, 0x68, 0xb3, 0xdc, 0xf5, 0xdf, 0xd8, 0x93, 0xef,
    0xbd, 0x07, 0xec, 0x60, 0x87, 0x54, 0xec, 0x14, 0x16, 0xa4, 0x79, 0xb7,
    0xab, 0xd0, 0x7e, 0x63, 0x72, 0x99, 0xe8, 0x33, 0x12, 0xe1, 0x1c, 0xdd,
    0xbe, 0xff, 0xc9, 0x3f, 0x2b, 0xb5, 0xbf, 0x06, 0x4d, 0x8e, 0xc2, 0xb1,
    0xd7, 0xfb, 0x2f, 0x2d, 0x80, 0x6a, 0x9c, 0xb0, 0x55, 0xdf, 0x85, 0x2f,
    0x08, 0x89, 0x62, 0x1e, 0xca, 0xb1, 0xbd, 0x9a, 0x1b, 0xd2, 0xb6, 0x74,
    0xcd, 0x8e, 0x1e, 0xf2, 0xad, 0x58, 0xd6, 0xa5, 0x69, 0x96, 0xed, 0xf5,
    0x19, 0xf0, 0x7c, 0xf1, 0xb6, 0x08, 0x23, 0xae, 0x61, 0x7c, 0x9d, 0x43,
    0x46, 0x90, 0x4f, 0xaf, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x82, 0x01,
    0x00, 0x23, 0x14, 0xe8, 0x9c, 0x1f, 0xf9, 0x19, 0x6c, 0x73, 0xf8, 0x6b,
    0xb8, 0xe1, 0x9a, 0x5f, 0x7d, 0x48, 0x9a, 0x1a, 0xaf, 0xb2, 0x42, 0xe8,
    0x16, 0xd4, 0x3c, 0x62, 0x1a, 0x37, 0xe2, 0xd9, 0x5f, 0xd3, 0xed, 0x5e,
    0x53, 0x91, 0xf3, 0xa3, 0x2b, 0x5a, 0x62, 0x30, 0xee, 0x64, 0x18, 0xf2,
    0x04, 0xa7, 0xfb, 0x11, 0x6d, 0xf2, 0xdd, 0x69, 0xfc, 0xef, 0xd2, 0x0e,
    0x6a, 0x39, 0xa1, 0x6f, 0x5f, 0x2f, 0x24, 0xbd, 0xf7, 0xa5, 0xdb, 0xdd,
    0x44, 0xc4, 0xfb, 0x3e, 0xa1, 0x36, 0xe5, 0x88, 0x97, 0x7a, 0x10, 0xce,
    0x70, 0xfb, 0x3d, 0xf5, 0xcc, 0x7e, 0xb9, 0x03, 0x71, 0x71, 0x27, 0x3d,
    0xe5, 0x44, 0x03, 0x0e, 0xe3, 0xfc, 0xcb, 0x42, 0xeb, 0xdb, 0x28, 0xd4,
    0x06, 0x3a, 0x1c, 0x9f, 0xa2, 0xb7, 0xea, 0xab, 0x97, 0x19, 0xc6, 0x1c,
    0xae, 0xcb, 0xf3, 0xc4, 0x3e, 0x27, 0xca, 0xdd, 0x6b, 0x00, 0xb6, 0xad,
    0x07, 0xf7, 0x04, 0x5a, 0xb4, 0x06, 0x3d, 0x7d, 0x08, 0x99, 0x68, 0xdd,
    0x62, 0x2d, 0x93, 0x76, 0x21, 0xfe, 0x31, 0xa1, 0xeb, 0xcc, 0x20, 0x11,
    0xdb, 0x5b, 0x03, 0x3c, 0xf2, 0x60, 0x29, 0x2c, 0x1f, 0xda, 0x3c, 0x31,
    0x07, 0xf1, 0x0f, 0xce, 0x08, 0xa0, 0x62, 0x6d, 0x5a, 0x33, 0xf2, 0xd4,
    0x16, 0x25, 0x0e, 0xb6, 0xf3, 0x23, 0xdd, 0xf4, 0xff, 0xf5, 0xab, 0x13,
    0x35, 0x62, 0xd7, 0x06, 0xfc, 0xaf, 0xa9, 0x74, 0xc1, 0x9a, 0x67, 0xb1,
    0xf0, 0xfa, 0xeb, 0xed, 0x63, 0x2a, 0x82, 0x17, 0xd1, 0xeb, 0x88, 0xa6,
    0x09, 0x97, 0xf9, 0x4b, 0x00, 0x49, 0x70, 0x5f, 0x35, 0x25, 0x48, 0xae,
    0xf2, 0xa0, 0xde, 0xb6, 0xb9, 0x62, 0x6d, 0x64, 0x77, 0x05, 0x64, 0xcb,
    0x56, 0xdc, 0xd2, 0xbf, 0x16, 0xe7, 0x10, 0x80, 0x52, 0x4f, 0x04, 0x85,
    0x31, 0x22, 0x1c, 0x1a, 0x71, 0x02, 0x81, 0x81, 0x00, 0xf3, 0x16, 0xe7,
    0xf4, 0x9c, 0x1a, 0xfb, 0x23, 0xe3, 0x63, 0x5a, 0x2e, 0x51, 0x7b, 0xb0,
    0x68, 0xe2, 0xa8, 0x9d, 0x1a, 0x6e, 0x43, 0x9e, 0x22, 0x47, 0x01, 0x48,
    0x91, 0xd2, 0xbf, 0xde, 0x30, 0x91, 0xbb, 0x23, 0xa2, 0x25, 0xd3, 0x96,
    0x7d, 0xc6, 0x22, 0x2f, 0xf6, 0x18, 0x99, 0x63, 0x5a, 0xf5, 0x69, 0xc0,
    0x44, 0x43, 0x4b, 0x21, 0xd4, 0xd2, 0x14, 0x22, 0xb1, 0xf9, 0x8d, 0xde,
    0xa9, 0x20, 0x68, 0x7c, 0x38, 0x28, 0xab, 0x7a, 0x77, 0x08, 0x19, 0x80,
    0x52, 0xff, 0xfa, 0xe5, 0xd3, 0xba, 0x33, 0x92, 0x4c, 0x4d, 0x6a, 0x4f,
    0xf3, 0x08, 0xf6, 0x75, 0x20, 0xc1, 0xca, 0xe9, 0x1d, 0xb4, 0xd3, 0x56,
    0xd6, 0x51, 0x32, 0x75, 0x6c, 0x07, 0x4b, 0x4c, 0x15, 0x8f, 0x63, 0x79,
    0x27, 0xcb, 0x79, 0x9c, 0x03, 0xd3, 0xf5, 0x90, 0xb5, 0xee, 0xce, 0x92,
    0xac, 0x07, 0x12, 0x89, 0xa9, 0x02, 0x81, 0x81, 0x00, 0xea, 0x3b, 0x62,
    0x00, 0xae, 0xa7, 0x5d, 0x56, 0x67, 0x76, 0x31, 0xf8, 0xda, 0x59, 0x52,
    0x82, 0x53, 0x7f, 0x49, 0x7c, 0x4a, 0x69, 0x4c, 0xd8, 0x4a, 0x35, 0x34,
    0x26, 0x86, 0x0a, 0xdf, 0xf3, 0x97, 0xae, 0xb1, 0x6d, 0x20, 0x85, 0xbd,
    0x20, 0x49, 0x2c, 0xa3, 0x11, 0xef, 0x76, 0x38, 0xe3, 0x7e, 0xf7, 0x36,
    0x8a, 0xa2, 0x79, 0xb8, 0xbc, 0xd1, 0xad, 0xf1, 0xbc, 0xc4, 0x5a, 0xac,
    0xf8, 0x43, 0xd7, 0x20, 0x28, 0xd9, 0xa0, 0xfa, 0xce, 0x8a, 0xfd, 0xee,
    0x53, 0x22, 0x13, 0xa5, 0x3a, 0x5f, 0xd4, 0x11, 0xe4, 0xf3, 0xf6, 0x78,
    0x1e, 0xa6, 0xab, 0x3e, 0xf9, 0xe9, 0xae, 0x52, 0xf9, 0x8d, 0xf5, 0x30,
    0x6e, 0xea, 0x7b, 0x39, 0x30, 0x0a, 0x4f, 0x0c, 0x65, 0x8e, 0x79, 0x14,
    0x13, 0xa9, 0x36, 0xe9, 0x68, 0x5e, 0x75, 0x8d, 0x47, 0x77, 0x4a, 0x5a,
    0xfc, 0xe7, 0x0c, 0x55, 0x97, 0x02, 0x81, 0x81, 0x00, 0xe1, 0xc8, 0x49,
    0xc3, 0x79, 0xe6, 0xd9, 0x92, 0x98, 0xd6, 0x23, 0xe1, 0xd7, 0x8f, 0x01,
    0xe8, 0x39, 0x76, 0x11, 0x0d, 0x86, 0x92, 0xc9, 0x62, 0xde, 0xfe, 0x83,
    0x2f, 0xe6, 0x46, 0x8d, 0x92, 0x21, 0x19, 0x67, 0x5c, 0x68, 0xb3, 0xe1,
    0x18, 0x8b, 0x1a, 0x1f, 0xdf, 0xfd, 0x33, 0x23, 0x04, 0x4d, 0xe3, 0x5e,
    0x50, 0x49, 0x0c, 0x94, 0xac, 0xbe, 0x3f, 0x33, 0x86, 0x88, 0xda, 0xcb,
    0x8f, 0xdc, 0xe9, 0x0f, 0x4f, 0x7d, 0xfe, 0xdf, 0xad, 0x85, 0x59, 0x62,
    0x46, 0x35, 0x3a, 0x0d, 0x93, 0x9d, 0x63, 0xdf, 0x23, 0x7d, 0xd8, 0x1b,
    0xd0, 0x09, 0x68, 0x18, 0x9c, 0x47, 0x43, 0x37, 0x48, 0x7f, 0x36, 0xb8,
    0x76, 0x31, 0x4e, 0xa3, 0x9f, 0xa6, 0x1a, 0xba, 0xdd, 0x5f, 0xa4, 0x2d,
    0xd8, 0xd7, 0x75, 0xe9, 0x81, 0xe8, 0xb2, 0x32, 0x29, 0xb9, 0x58, 0x27,
    0x05, 0x1f, 0x40, 0x89, 0x99, 0x02, 0x81, 0x80, 0x03, 0x11, 0xf2, 0x2f,
    0xe6, 0x1a, 0x25, 0x7c, 0x0b, 0x71, 0x1e, 0x28, 0x8f, 0xdc, 0x19, 0xdb,
    0xc1, 0x72, 0x6a, 0xa9, 0x29, 0x64, 0x85, 0xf0, 0x53, 0xd0, 0xac, 0xb5,
    0x50, 0x8e, 0x75, 0xef, 0x00, 0x6e, 0xf6, 0xbe, 0x62, 0x6d, 0xbe, 0x53,
    0xdd, 0x01, 0xab, 0xc5, 0x81, 0x69, 0x00, 0x47, 0xe3, 0x13, 0x24, 0x70,
    0xde, 0xbe, 0x4f, 0xb0, 0x91, 0xe3, 0x00, 0xc7, 0x23, 0x9f, 0xb8, 0xa2,
    0xac, 0x68, 0xf6, 0x6f, 0xf2, 0xa2, 0x4c, 0x87, 0x46, 0x85, 0x1b, 0x89,
    0x95, 0x69, 0xc2, 0x81, 0xa8, 0x5d, 0xe4, 0x3a, 0x27, 0x04, 0x75, 0x00,
    0x99, 0xd8, 0xa6, 0xc3, 0xa9, 0xcf, 0x72, 0x20, 0xf9, 0x3e, 0x5c, 0x7b,
    0x27, 0xb1, 0xa3, 0xc7, 0x40, 0x6b, 0x56, 0xd3, 0x41, 0xf1, 0xce, 0x54,
    0x8f, 0xa4, 0x3b, 0x9c, 0xf5, 0x97, 0xfe, 0xd9, 0x0f, 0xf4, 0x36, 0x8b,
    0x18, 0x10, 0x2e, 0x53, 0x02, 0x81, 0x80, 0x72, 0xc7, 0x72, 0x18, 0x07,
    0xe4, 0x0b, 0x07, 0x1e, 0xa6, 0x4a, 0x7b, 0x10, 0x6f, 0xdf, 0xfc, 0xf5,
    0xd6, 0x1d, 0x6d, 0x7e, 0xb4, 0xab, 0x14, 0x5d, 0x9c, 0x2f, 0x39, 0x14,
    0x5c, 0xd1, 0x6a, 0x4f, 0xbe, 0x06, 0x4c, 0xa5, 0xcb, 0x48, 0x3a, 0xee,
    0x46, 0xb4, 0xd7, 0x4a, 0xea, 0xef, 0x40, 0x94, 0x1b, 0xe1, 0x7f, 0x19,
    0xa0, 0x3a, 0xd8, 0xb4, 0x75, 0x9f, 0x2c, 0x46, 0x85, 0x26, 0x53, 0x97,
    0x1e, 0x6f, 0xb4, 0xb0, 0x55, 0x1f, 0x76, 0x14, 0xd2, 0xc5, 0x0b, 0xe2,
    0x2b, 0x48, 0x50, 0x73, 0x1b, 0x43, 0x88, 0x5e, 0xee, 0x5e, 0xdc, 0x2a,
    0x5d, 0x40, 0x5b, 0x94, 0xa1, 0xd7, 0xab, 0xec, 0x24, 0x76, 0xf8, 0x8a,
    0xee, 0x80, 0xcd, 0x22, 0x20, 0x93, 0x2a, 0x73, 0x08, 0x85, 0x47, 0x13,
    0xe0, 0x6c, 0x08, 0x5b, 0xa3, 0xb8, 0x04, 0x5e, 0x12, 0xd1, 0xef, 0x49,
    0xda, 0xf6, 0xbd};

// kExampleRSAPSSKeyPKCS8 contains a DER-encoded RSASSA-PSS-params:
//    Hash Algorithm: sha256
//    Mask Algorithm: mgf1 with sha256
static const uint8_t kExampleRSAPSSKeyPKCS8[] = {
    0x30, 0x82, 0x04, 0xea, 0x02, 0x01, 0x00, 0x30, 0x38, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x2b, 0xa0, 0x0d,
    0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02,
    0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
    0x65, 0x03, 0x04, 0x02, 0x01, 0x04, 0x82, 0x04, 0xa9, 0x30, 0x82, 0x04,
    0xa5, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01, 0x01, 0x00, 0xd6, 0xe6, 0x38,
    0xa9, 0x70, 0x5c, 0x90, 0xf5, 0xd6, 0xaf, 0x78, 0xbb, 0x00, 0x05, 0xf3,
    0x7a, 0x6a, 0x3c, 0x0c, 0x12, 0x56, 0x32, 0xb6, 0xf5, 0x91, 0xbd, 0xad,
    0x3b, 0x22, 0x06, 0xbb, 0x3e, 0xc6, 0x01, 0x89, 0x14, 0xed, 0xcc, 0x0e,
    0xa5, 0xb1, 0x80, 0x69, 0x6b, 0x23, 0x83, 0xf2, 0x1f, 0x1c, 0x1c, 0x62,
    0x9d, 0x30, 0x61, 0x94, 0x45, 0x25, 0xf0, 0xf6, 0xe3, 0xb0, 0x84, 0xdc,
    0x5c, 0x80, 0x3d, 0xcb, 0xd6, 0x4c, 0xd7, 0x3f, 0x65, 0x1f, 0x72, 0x15,
    0xf9, 0x36, 0xa9, 0x70, 0x80, 0x65, 0xce, 0x92, 0xfb, 0x79, 0xa1, 0x39,
    0x67, 0xf9, 0xb4, 0x7f, 0x5b, 0xac, 0x74, 0xd2, 0x1d, 0x11, 0xb3, 0x58,
    0x37, 0x58, 0xae, 0xfc, 0x84, 0xd0, 0xe7, 0xe8, 0x96, 0xab, 0xcc, 0x9c,
    0x60, 0x01, 0x25, 0x79, 0x18, 0x9f, 0xa8, 0xbb, 0x37, 0xa8, 0x30, 0x56,
    0x01, 0x40, 0xe4, 0x28, 0x4d, 0x75, 0x0a, 0x21, 0x7c, 0xf4, 0xe1, 0x5e,
    0xd4, 0x44, 0x87, 0x55, 0xf8, 0xa7, 0xc8, 0xc3, 0x36, 0x30, 0x18, 0xc8,
    0xbe, 0x6e, 0x1a, 0x2e, 0x08, 0x36, 0x44, 0xa2, 0x68, 0xf8, 0x7e, 0xb7,
    0xe2, 0xd3, 0x7b, 0xcf, 0x5e, 0xd1, 0x8b, 0x17, 0x7b, 0x78, 0x75, 0x75,
    0x99, 0x35, 0x1e, 0x4a, 0xbc, 0x48, 0x3d, 0xb7, 0x25, 0x1b, 0xcc, 0xf1,
    0x1c, 0xbf, 0x8b, 0x9a, 0xd9, 0xba, 0xca, 0x02, 0x7f, 0x69, 0x8e, 0xbe,
    0xed, 0xd7, 0x4d, 0x8e, 0xfa, 0x8c, 0x76, 0x43, 0xe3, 0x2b, 0xa8, 0xbe,
    0xb1, 0xf2, 0x24, 0xd3, 0xf0, 0x34, 0x30, 0xd4, 0x8e, 0x85, 0x3c, 0x68,
    0xea, 0xf7, 0x0a, 0x44, 0xcf, 0x66, 0xb0, 0xa9, 0x28, 0x54, 0x19, 0x00,
    0xb5, 0x7f, 0xce, 0x65, 0xa3, 0xae, 0xce, 0xd7, 0x20, 0x88, 0x96, 0xc1,
    0xe9, 0x62, 0x02, 0xab, 0x15, 0x3d, 0x70, 0x34, 0xb9, 0xf9, 0xa5, 0x92,
    0xdf, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02, 0x82, 0x01, 0x00, 0x77, 0xd2,
    0xba, 0x79, 0x4f, 0x9b, 0x6d, 0x80, 0x86, 0x0a, 0x25, 0x29, 0x2f, 0xa4,
    0x21, 0xd4, 0x79, 0xf3, 0xad, 0x4f, 0x74, 0x1c, 0xa7, 0x03, 0x5a, 0x16,
    0x06, 0xc7, 0x57, 0xfb, 0xa4, 0xe0, 0x25, 0x8b, 0xa9, 0xa1, 0xaf, 0xcd,
    0xfd, 0xc4, 0xc9, 0xaa, 0x2e, 0x1d, 0x7d, 0x1a, 0x0f, 0x20, 0xe5, 0x8d,
    0xb6, 0x36, 0xab, 0x49, 0x5d, 0x61, 0x08, 0xfc, 0xc8, 0xcc, 0xc0, 0x19,
    0xf4, 0xbc, 0xe7, 0x0c, 0x8d, 0xe1, 0x99, 0xca, 0xd3, 0x20, 0x51, 0x5e,
    0x4d, 0x7e, 0x3d, 0xca, 0x50, 0x5e, 0x67, 0x65, 0xbd, 0x72, 0xd6, 0xa8,
    0xdc, 0xd7, 0x6a, 0xd6, 0x07, 0xc5, 0x57, 0xed, 0x73, 0x52, 0x9e, 0x11,
    0xff, 0xf8, 0x2c, 0x4c, 0xb5, 0xa4, 0x60, 0xcf, 0x4e, 0x52, 0xe4, 0xcc,
    0x1b, 0xcd, 0x34, 0x16, 0x6e, 0xe5, 0x19, 0xf0, 0xfd, 0x22, 0xd5, 0x76,
    0xce, 0x1d, 0xb8, 0xce, 0xfe, 0xe0, 0xdb, 0x74, 0x19, 0xa9, 0x27, 0xb6,
    0x30, 0x9e, 0x63, 0xa9, 0xf8, 0x2e, 0x06, 0xae, 0xc5, 0x85, 0x32, 0x60,
    0xa0, 0xb0, 0x7c, 0x35, 0x65, 0xe5, 0xd5, 0xdc, 0xfc, 0x97, 0xd1, 0x57,
    0xb8, 0x84, 0xd4, 0xbf, 0x1d, 0x0b, 0x6c, 0xe9, 0x2c, 0x57, 0xeb, 0x5c,
    0x51, 0x12, 0x76, 0x0e, 0x57, 0x9d, 0x6c, 0x44, 0xe8, 0x5a, 0xfe, 0x1b,
    0x66, 0xb5, 0x38, 0x72, 0x31, 0x15, 0x91, 0xe6, 0xc5, 0xb6, 0xad, 0xf0,
    0x89, 0x31, 0xdb, 0x88, 0x2c, 0x52, 0xf8, 0x3f, 0x72, 0x2c, 0x4f, 0x9a,
    0x93, 0x55, 0xd4, 0xf4, 0xe2, 0x34, 0x73, 0xf0, 0x07, 0x34, 0xe6, 0xbf,
    0x70, 0x94, 0x73, 0x5f, 0xe1, 0x3a, 0xe3, 0x37, 0x3f, 0xaf, 0x38, 0x24,
    0xbd, 0x12, 0x2f, 0xba, 0xcc, 0x89, 0x15, 0xe0, 0xcd, 0x43, 0x27, 0x8c,
    0xbd, 0x3e, 0x52, 0xac, 0x08, 0x86, 0x32, 0x81, 0x0d, 0xe4, 0x9d, 0x83,
    0xfa, 0x11, 0x02, 0x81, 0x81, 0x00, 0xf9, 0xae, 0x0e, 0xed, 0xb1, 0x57,
    0xf1, 0x90, 0xa5, 0xeb, 0x5d, 0x08, 0x7d, 0x1c, 0xe2, 0x4c, 0xab, 0x15,
    0xa6, 0x7d, 0xb2, 0xa1, 0x01, 0x80, 0x97, 0xb0, 0x3f, 0x1d, 0x02, 0x19,
    0xa0, 0xb0, 0x61, 0xd7, 0xff, 0x23, 0x88, 0xa3, 0x32, 0xe7, 0xbf, 0xfe,
    0xf7, 0x7d, 0xe4, 0x5e, 0xed, 0xa2, 0xe2, 0x52, 0xb6, 0xa6, 0x13, 0x83,
    0x12, 0x4e, 0xf5, 0x33, 0xf6, 0x33, 0xf4, 0x0c, 0x90, 0x97, 0x7a, 0x0a,
    0xac, 0x12, 0xa5, 0x80, 0x16, 0x5c, 0x96, 0x60, 0xed, 0x78, 0xd2, 0x77,
    0x8f, 0xc8, 0xcd, 0xe5, 0x0d, 0x55, 0x2b, 0x13, 0xbf, 0xe0, 0x48, 0xbe,
    0xe0, 0xd1, 0x01, 0x3a, 0x4d, 0xe3, 0x9d, 0x9c, 0xe5, 0xe2, 0xd9, 0xc9,
    0xd1, 0x39, 0x90, 0xed, 0xdc, 0xcd, 0x10, 0x16, 0xf4, 0x92, 0x41, 0x12,
    0xd2, 0x4a, 0x78, 0x81, 0x47, 0xe5, 0x4b, 0x46, 0x55, 0x97, 0x7b, 0xab,
    0xc2, 0xbb, 0x02, 0x81, 0x81, 0x00, 0xdc, 0x56, 0xc8, 0x4e, 0x1a, 0x0d,
    0xd4, 0xbf, 0x52, 0xb8, 0xc0, 0xae, 0xd3, 0x79, 0x41, 0x20, 0xb9, 0x87,
    0x19, 0x38, 0xc8, 0x19, 0x2a, 0xfe, 0x6f, 0xcd, 0xe3, 0x0f, 0x56, 0x0d,
    0x97, 0x57, 0x1d, 0xd6, 0xbe, 0xf0, 0x19, 0xc0, 0xf2, 0x5e, 0xd8, 0x15,
    0xab, 0xf2, 0x36, 0x62, 0xfd, 0x59, 0x09, 0xc3, 0x75, 0x2e, 0xcc, 0xf2,
    0xb3, 0x52, 0x91, 0x31, 0xc9, 0xb9, 0x1a, 0xe0, 0x15, 0x61, 0x1f, 0x1e,
    0x22, 0x39, 0x4d, 0x9d, 0x62, 0xae, 0x6f, 0x3c, 0x65, 0xe6, 0x4b, 0x30,
    0x4b, 0x40, 0xae, 0x97, 0x68, 0x7f, 0x1c, 0x31, 0x8e, 0xa8, 0x00, 0x68,
    0x59, 0xef, 0x5d, 0xc2, 0x7b, 0x40, 0x8c, 0xf0, 0x9a, 0x28, 0xb6, 0xbd,
    0x80, 0x72, 0xe3, 0xe6, 0x3c, 0x39, 0xbb, 0x9c, 0x3f, 0x2b, 0xad, 0x23,
    0xd0, 0x28, 0x72, 0x88, 0x7f, 0x2e, 0x21, 0x9c, 0x69, 0x37, 0xd9, 0x73,
    0x88, 0x2d, 0x02, 0x81, 0x81, 0x00, 0xc7, 0xcb, 0x84, 0x00, 0xaa, 0xa8,
    0x20, 0x12, 0xca, 0xe2, 0xb3, 0x98, 0xa6, 0x5d, 0x65, 0xad, 0x4a, 0x3e,
    0x51, 0xf3, 0x45, 0xec, 0xd7, 0x01, 0xea, 0xe0, 0x4b, 0x30, 0xd0, 0x4c,
    0x67, 0xac, 0x9d, 0x2f, 0x35, 0xe8, 0x5e, 0x5f, 0x91, 0xcd, 0xa7, 0x72,
    0xc4, 0xe3, 0x45, 0xca, 0xbc, 0xc8, 0x59, 0xe9, 0x2c, 0x65, 0x6b, 0x39,
    0xb8, 0x7d, 0xdc, 0x73, 0x4f, 0x31, 0xb5, 0xba, 0x14, 0x3e, 0x52, 0x97,
    0x95, 0xa8, 0x7a, 0x2d, 0x2c, 0x37, 0xbc, 0xd2, 0x25, 0x55, 0x48, 0xbe,
    0x0a, 0x1d, 0xfa, 0xbc, 0xae, 0xd5, 0x97, 0x59, 0xff, 0x71, 0xdf, 0x64,
    0x77, 0x0c, 0x68, 0xba, 0x1a, 0x37, 0xe9, 0x18, 0xf3, 0x50, 0x7b, 0x4c,
    0xa7, 0x13, 0x34, 0xa9, 0x52, 0xc5, 0xf9, 0x79, 0x15, 0x0e, 0xaf, 0xe9,
    0xfa, 0xc3, 0x42, 0xf8, 0xd8, 0xea, 0x60, 0x39, 0x11, 0x26, 0xdd, 0xfc,
    0xcd, 0x1f, 0x02, 0x81, 0x81, 0x00, 0xbf, 0x70, 0x53, 0x36, 0xb0, 0xa2,
    0x65, 0xbf, 0x19, 0x73, 0x56, 0x4e, 0x4f, 0x84, 0xe1, 0x96, 0x3a, 0x3b,
    0xec, 0x29, 0x0e, 0x89, 0xf7, 0x75, 0xfb, 0xd4, 0x29, 0xbc, 0xd0, 0x6a,
    0x9c, 0x9c, 0x5f, 0x5b, 0x8e, 0xa7, 0xf3, 0xe8, 0x55, 0x6d, 0x1c, 0x57,
    0xaf, 0xfe, 0x91, 0x4c, 0x6e, 0x8c, 0xa6, 0xd4, 0x01, 0x03, 0x79, 0xa4,
    0x41, 0x33, 0xb2, 0xdd, 0x4c, 0x44, 0x1c, 0xda, 0xa7, 0x18, 0xac, 0x3d,
    0xb5, 0x44, 0x8e, 0x6f, 0xf9, 0x26, 0x4f, 0x95, 0x6d, 0x7f, 0xaa, 0xc2,
    0x3a, 0x0f, 0xe7, 0xe1, 0xbb, 0x33, 0xfd, 0x5d, 0xfd, 0x71, 0x70, 0x71,
    0x39, 0xfa, 0xa4, 0x3e, 0xee, 0x71, 0x86, 0x43, 0x21, 0xdd, 0xcf, 0xf0,
    0x54, 0x22, 0xfe, 0x69, 0xb2, 0xd8, 0x02, 0x6e, 0x5d, 0xe0, 0x67, 0xdd,
    0x5e, 0xe5, 0x55, 0xa5, 0xcc, 0xc2, 0x5b, 0x07, 0x97, 0xcf, 0xa3, 0x8a,
    0x22, 0xbd, 0x02, 0x81, 0x81, 0x00, 0xee, 0x18, 0x67, 0x30, 0x7d, 0x7d,
    0xd1, 0x5b, 0xef, 0x28, 0xfe, 0x1e, 0xf9, 0x27, 0x07, 0x7b, 0x3b, 0xcf,
    0xb1, 0xa2, 0x62, 0xe6, 0xd2, 0x04, 0x19, 0xb3, 0x0d, 0xfe, 0xb6, 0xc4,
    0x5d, 0x56, 0x71, 0xb7, 0x85, 0xea, 0x24, 0x69, 0x73, 0x36, 0x10, 0xf2,
    0x45, 0x18, 0x1d, 0xa0, 0xc9, 0x5b, 0x59, 0xbb, 0xd5, 0xe0, 0x7c, 0x52,
    0x73, 0x78, 0x46, 0x61, 0xf2, 0x0d, 0x6a, 0xe0, 0xf0, 0x62, 0xac, 0xc0,
    0x63, 0xcc, 0xbe, 0x61, 0xa2, 0x6a, 0x06, 0xeb, 0x30, 0x41, 0x93, 0xa6,
    0x5d, 0xae, 0xf8, 0x8d, 0xf8, 0x0b, 0xca, 0x7f, 0x8f, 0x76, 0xef, 0x4a,
    0x3a, 0x37, 0x80, 0x9e, 0x17, 0x27, 0x6b, 0xb2, 0x5c, 0x37, 0x0f, 0x42,
    0x20, 0x5f, 0xaf, 0xee, 0x4c, 0x55, 0xeb, 0x6d, 0x44, 0xe0, 0x8a, 0x94,
    0xe5, 0x7a, 0xa5, 0x8c, 0x31, 0xe5, 0xae, 0xe3, 0x27, 0xa7, 0x8e, 0xa8,
    0x94, 0xc2};

// badRSAPSSKeyPKCS8_SaltLengthTooLarge is encoded in a PKCS #8 PrivateKeyInfo.
// badRSAPSSKeyPKCS8_SaltLengthTooLarge contains a DER-encoded RSASSA-PSS-params:
//    Hash Algorithm: sha256
//    Mask Algorithm: mgf1 with sha256
//    Salt Length: 511
static const uint8_t badRSAPSSKeyPKCS8_SaltLengthTooLarge[] = {
    0x30, 0x82, 0x04, 0xed, 0x02, 0x01, 0x00, 0x30, 0x3e, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x0a, 0x30, 0x31, 0xa0, 0x0d,
    0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01, 0x65, 0x03, 0x04, 0x02,
    0x01, 0xa1, 0x1a, 0x30, 0x18, 0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7,
    0x0d, 0x01, 0x01, 0x08, 0x30, 0x0b, 0x06, 0x09, 0x60, 0x86, 0x48, 0x01,
    0x65, 0x03, 0x04, 0x02, 0x01, 0xa2, 0x04, 0x02, 0x02, 0x01, 0xff, 0x04,
    0x82, 0x04, 0xa6, 0x30, 0x82, 0x04, 0xa2, 0x02, 0x01, 0x00, 0x02, 0x82,
    0x01, 0x01, 0x00, 0xc4, 0xd8, 0xc8, 0x45, 0x0c, 0x9a, 0xc6, 0x5a, 0xf0,
    0xe8, 0x34, 0x3b, 0x9d, 0x73, 0x01, 0xe0, 0x71, 0xb5, 0x3a, 0x55, 0x9e,
    0xc6, 0xcd, 0xdc, 0xdb, 0x98, 0x29, 0xea, 0x75, 0x4f, 0xf9, 0xa9, 0x5c,
    0xc1, 0x40, 0x1d, 0x7c, 0x7d, 0x60, 0x8f, 0x1e, 0x1d, 0xde, 0xb1, 0x9b,
    0xca, 0x8d, 0x7d, 0x83, 0x09, 0xce, 0x56, 0x94, 0x22, 0x5e, 0xce, 0xa3,
    0x5f, 0x66, 0x3e, 0x24, 0x08, 0x78, 0x05, 0xc1, 0xe2, 0x11, 0x08, 0x0d,
    0x58, 0x1c, 0xec, 0xd3, 0xfd, 0xdb, 0x36, 0x39, 0x74, 0x7d, 0x79, 0xef,
    0xd2, 0x32, 0x99, 0xb2, 0x44, 0xa0, 0x77, 0xf8, 0x88, 0x7e, 0x16, 0x9b,
    0x39, 0x40, 0x48, 0x48, 0x4f, 0x66, 0x26, 0xc9, 0xee, 0x7a, 0xdd, 0xfa,
    0xf8, 0x74, 0xf6, 0xdb, 0xbc, 0x57, 0x5a, 0xc3, 0x4e, 0xd2, 0xb3, 0x5d,
    0xe6, 0xb9, 0x6e, 0x63, 0x72, 0x46, 0xde, 0xe4, 0xff, 0x60, 0x9f, 0x00,
    0x0b, 0x2b, 0x6b, 0xf2, 0xc9, 0x41, 0x0d, 0x0e, 0x99, 0x62, 0xda, 0xeb,
    0x52, 0x5f, 0xfd, 0x37, 0x04, 0xae, 0x56, 0x48, 0x44, 0x6f, 0x2c, 0x86,
    0x06, 0xce, 0x4c, 0x19, 0x84, 0x86, 0x73, 0x0c, 0x9a, 0xb6, 0x89, 0x0e,
    0x42, 0x81, 0xe8, 0xcf, 0x25, 0x3a, 0x4c, 0x3b, 0xd4, 0xaf, 0xec, 0xaf,
    0x99, 0x5c, 0x27, 0x7c, 0x02, 0x4a, 0xdb, 0x1d, 0x52, 0x00, 0x48, 0x4c,
    0x6b, 0x06, 0x9b, 0x2a, 0xaa, 0x38, 0xc0, 0xac, 0x3d, 0xf9, 0xc2, 0x06,
    0x2b, 0x18, 0xc5, 0x48, 0x56, 0x3f, 0xe4, 0xab, 0x5a, 0x2b, 0x3e, 0x87,
    0x3d, 0x29, 0xbb, 0x93, 0x05, 0x6c, 0xcd, 0xaa, 0xd0, 0x0a, 0x13, 0x2f,
    0x96, 0xcd, 0xe4, 0xa8, 0xa4, 0x7b, 0xc1, 0x39, 0x1d, 0x27, 0xc3, 0x20,
    0xa0, 0xc0, 0xe5, 0x05, 0xc6, 0x6e, 0x8e, 0x46, 0x77, 0x46, 0xfb, 0xdc,
    0xbb, 0x1f, 0x3c, 0x6e, 0xc8, 0xd3, 0xdb, 0x02, 0x03, 0x01, 0x00, 0x01,
    0x02, 0x82, 0x01, 0x00, 0x30, 0x00, 0xe5, 0xa5, 0xe1, 0x13, 0x05, 0x78,
    0x3b, 0xf3, 0x84, 0x79, 0x5d, 0x46, 0x24, 0x99, 0x05, 0x61, 0xba, 0x9e,
    0x12, 0xa0, 0x4c, 0xa7, 0xa5, 0x6d, 0x90, 0x1b, 0x3b, 0x0c, 0x8b, 0xc7,
    0xbf, 0x74, 0xe9, 0x1b, 0xc7, 0xb6, 0x13, 0x8c, 0x13, 0x3e, 0x19, 0x86,
    0x9a, 0x85, 0x64, 0xc7, 0xcc, 0xe2, 0xe1, 0x4c, 0xbf, 0xc7, 0x39, 0xf3,
    0x8e, 0x67, 0xbf, 0x59, 0x3c, 0x14, 0x4c, 0xa0, 0xaa, 0xcd, 0x64, 0x75,
    0xe0, 0x3b, 0x32, 0xa2, 0x96, 0x77, 0xfb, 0xb3, 0x57, 0x83, 0x8e, 0xed,
    0xe3, 0x8c, 0x11, 0x06, 0xe6, 0x30, 0x52, 0x29, 0xaf, 0x94, 0x29, 0x9b,
    0x7d, 0xb3, 0x12, 0x78, 0x0e, 0x1a, 0x3e, 0xc1, 0x0a, 0xd1, 0x7b, 0x69,
    0x12, 0xc0, 0xf0, 0x55, 0x1a, 0xf4, 0xed, 0x20, 0x97, 0x2c, 0x6b, 0xb9,
    0x8e, 0x16, 0xab, 0xf5, 0xbd, 0x8e, 0xaa, 0xb5, 0xc4, 0x86, 0x7a, 0x9b,
    0x1d, 0x6f, 0xe3, 0xf8, 0x4f, 0x27, 0xde, 0x5d, 0x05, 0x9a, 0x4d, 0xa6,
    0xd7, 0xb2, 0x18, 0x54, 0xbe, 0x5c, 0x35, 0x4e, 0xb8, 0x62, 0xb5, 0x96,
    0xc8, 0xb3, 0x41, 0xae, 0x74, 0x51, 0x48, 0x45, 0xa1, 0xaa, 0xad, 0x64,
    0x3d, 0x2a, 0x78, 0xd0, 0xf4, 0x49, 0xcb, 0xd5, 0x24, 0x56, 0x23, 0x48,
    0x42, 0x20, 0x3f, 0xbe, 0x03, 0x91, 0xc1, 0xb8, 0x1a, 0xcc, 0x26, 0x0b,
    0xdc, 0x9f, 0x24, 0x75, 0x12, 0xf8, 0xd5, 0x0c, 0xbc, 0x84, 0xdc, 0x7a,
    0x94, 0xb8, 0x9d, 0x3a, 0x3c, 0x0e, 0xcf, 0x06, 0xfc, 0xf1, 0xe4, 0xf8,
    0x46, 0xc3, 0xa0, 0x8f, 0x09, 0x0d, 0xbc, 0xcc, 0x15, 0x0c, 0x4e, 0x92,
    0x21, 0xe6, 0x45, 0x5a, 0x5e, 0xe0, 0xbe, 0x8d, 0xab, 0x1d, 0x3c, 0x32,
    0xa7, 0x7c, 0x36, 0xda, 0xc9, 0x5a, 0xae, 0xd1, 0xf1, 0x0f, 0x4c, 0x2f,
    0xbf, 0x57, 0x0d, 0x7c, 0x70, 0x3a, 0x71, 0x11, 0x02, 0x81, 0x81, 0x00,
    0xfd, 0x50, 0xa3, 0xb7, 0x52, 0xfa, 0x64, 0x84, 0x50, 0x1a, 0x35, 0x8c,
    0xe0, 0xe9, 0x23, 0x8a, 0x47, 0xdb, 0x99, 0x4a, 0xa4, 0x22, 0xff, 0xa6,
    0x30, 0x03, 0x38, 0x2e, 0x11, 0x99, 0xeb, 0x27, 0x45, 0x96, 0x2f, 0xb6,
    0xae, 0x28, 0x82, 0xad, 0x32, 0xbd, 0x9a, 0xf8, 0xd2, 0x2c, 0xea, 0xe0,
    0x49, 0xd2, 0x3c, 0x17, 0x92, 0xf0, 0xe1, 0x0b, 0xb4, 0xec, 0xfd, 0xf2,
    0xac, 0x76, 0xa5, 0xf5, 0x89, 0x91, 0xe7, 0x0c, 0x7b, 0xe4, 0x5a, 0x02,
    0x88, 0xf9, 0xf7, 0x10, 0x40, 0x3b, 0x0a, 0x12, 0x83, 0x5c, 0x4b, 0x62,
    0x7b, 0xdb, 0xbe, 0x7f, 0x6b, 0x44, 0x81, 0xaa, 0xab, 0x6a, 0x2f, 0x72,
    0xf2, 0x72, 0xa2, 0x84, 0x3a, 0x6a, 0x38, 0x7b, 0x03, 0x16, 0x77, 0x4b,
    0xd8, 0x41, 0x44, 0xfe, 0xcb, 0xb1, 0x3c, 0x4c, 0x96, 0x1c, 0x01, 0x75,
    0x00, 0x09, 0xa2, 0xae, 0xe5, 0x7c, 0x1b, 0xff, 0x02, 0x81, 0x81, 0x00,
    0xc6, 0xee, 0xeb, 0x24, 0x69, 0x78, 0xda, 0x36, 0x9a, 0xb0, 0xf9, 0x11,
    0x57, 0xc3, 0x5a, 0x63, 0xa4, 0x57, 0x50, 0xc7, 0x1e, 0xf7, 0x46, 0xd0,
    0xa7, 0xac, 0x58, 0x9c, 0xdc, 0x93, 0x6d, 0x57, 0x78, 0xb7, 0x76, 0x88,
    0x65, 0x2f, 0x13, 0xb6, 0x8c, 0x2e, 0xb9, 0x73, 0xc4, 0x33, 0x63, 0x8e,
    0x70, 0xfb, 0x79, 0x68, 0x0d, 0x76, 0xf2, 0x6a, 0x01, 0x65, 0x07, 0x13,
    0x5d, 0x14, 0x4c, 0xd8, 0x62, 0x2d, 0x7e, 0xa0, 0x13, 0xbf, 0x94, 0x28,
    0xee, 0xb5, 0x60, 0x94, 0x8e, 0xfe, 0x4b, 0xdd, 0x6e, 0xef, 0xf8, 0xe0,
    0x80, 0x71, 0x18, 0x69, 0x44, 0xab, 0x74, 0xed, 0x36, 0x2c, 0x5a, 0xfc,
    0xc0, 0x5a, 0xbd, 0x21, 0x61, 0x44, 0x65, 0x31, 0x9e, 0x00, 0x2a, 0x3c,
    0x99, 0x06, 0x9c, 0x4d, 0xc3, 0x97, 0x09, 0x1c, 0xc0, 0xd5, 0xd2, 0x4e,
    0xf7, 0x7e, 0xa1, 0xd1, 0xa6, 0x47, 0x38, 0x25, 0x02, 0x81, 0x80, 0x01,
    0x4f, 0x70, 0x79, 0x5b, 0x49, 0x86, 0x49, 0x94, 0xec, 0x7d, 0xc8, 0x6b,
    0xc5, 0x68, 0xf4, 0xa7, 0x28, 0x80, 0xa3, 0x7d, 0x33, 0xdd, 0x24, 0xab,
    0xec, 0xe8, 0x56, 0x7c, 0xaa, 0xd2, 0x27, 0x92, 0xd9, 0x93, 0x07, 0x9b,
    0xe1, 0x03, 0xc4, 0x07, 0x96, 0x29, 0x7e, 0x0e, 0x00, 0x43, 0x7a, 0xc3,
    0x86, 0xfd, 0xde, 0x95, 0x58, 0xff, 0xa1, 0x02, 0xdf, 0x92, 0xf4, 0xb5,
    0x65, 0xab, 0xb8, 0x18, 0x6d, 0x13, 0xc7, 0xe1, 0xc0, 0x2d, 0xa3, 0x03,
    0xb8, 0x5f, 0x49, 0xcb, 0x3b, 0x42, 0xf3, 0x13, 0x90, 0xdf, 0xa6, 0xf3,
    0xc3, 0x10, 0x3a, 0x32, 0x80, 0xd9, 0x36, 0xdb, 0xca, 0x21, 0xd2, 0xb3,
    0x64, 0x5a, 0x68, 0xda, 0xb0, 0x7f, 0xf8, 0x4a, 0xec, 0xcb, 0xd3, 0x2d,
    0x9c, 0x30, 0xd2, 0x2c, 0x95, 0x8f, 0x1d, 0x4a, 0x4d, 0x7c, 0xc8, 0x71,
    0x26, 0x9b, 0x9b, 0xd1, 0xfe, 0x11, 0x5b, 0x02, 0x81, 0x80, 0x22, 0x7a,
    0xc0, 0x98, 0xc4, 0x0b, 0x25, 0x4f, 0x37, 0x8a, 0x9a, 0xf0, 0xcd, 0x94,
    0x57, 0x73, 0x53, 0xc5, 0xaa, 0x83, 0x88, 0xb3, 0x75, 0x3b, 0xaf, 0x04,
    0x29, 0x2b, 0xae, 0xf3, 0x82, 0x8d, 0x19, 0xa8, 0x3a, 0xcc, 0x33, 0x35,
    0x70, 0xa6, 0x40, 0x2f, 0xcd, 0x06, 0xea, 0xa7, 0xb9, 0x86, 0xb3, 0xc0,
    0x72, 0x6f, 0x7d, 0x7d, 0xf3, 0x50, 0x3a, 0x43, 0xa7, 0x71, 0x4c, 0xbd,
    0xba, 0xff, 0xfa, 0x2c, 0x46, 0x79, 0x3a, 0x53, 0x07, 0x3b, 0xfc, 0xfb,
    0xc4, 0xec, 0x9a, 0xc8, 0x2a, 0xa0, 0x0b, 0x9d, 0x5e, 0x36, 0x93, 0x2c,
    0x86, 0xfd, 0x9f, 0xe0, 0x53, 0xd8, 0xd5, 0x7d, 0xe4, 0x9b, 0x50, 0x3f,
    0xa9, 0xee, 0x42, 0x2c, 0x97, 0x99, 0x7c, 0xf8, 0x2c, 0x59, 0xea, 0x70,
    0x01, 0xe1, 0x5c, 0x98, 0x2e, 0x05, 0x24, 0xcb, 0x2a, 0xb3, 0x73, 0x24,
    0x28, 0xa9, 0xec, 0xb7, 0x05, 0x5d, 0x02, 0x81, 0x80, 0x51, 0x25, 0xc5,
    0xc4, 0x8c, 0xa1, 0xdc, 0x5a, 0x7f, 0xf2, 0xe3, 0xf7, 0x3f, 0xb8, 0x10,
    0x6e, 0xea, 0x71, 0x19, 0x24, 0x58, 0x85, 0xb8, 0x08, 0x06, 0x66, 0x86,
    0x70, 0x7f, 0xcb, 0x0e, 0xe4, 0xa8, 0xd8, 0x20, 0xf3, 0xd2, 0xa8, 0xaa,
    0xac, 0xe6, 0x31, 0x05, 0x91, 0x46, 0x23, 0xe3, 0xb9, 0x1c, 0xb8, 0x5e,
    0x46, 0x4f, 0xc6, 0x28, 0x13, 0x3f, 0xa6, 0xd5, 0x43, 0xe5, 0xa0, 0xab,
    0xa2, 0xce, 0xa9, 0x62, 0x6c, 0x39, 0x93, 0x75, 0x7b, 0xdc, 0x98, 0x36,
    0xec, 0x7c, 0x83, 0x27, 0xbe, 0x39, 0x99, 0x1d, 0xce, 0x44, 0x28, 0xcf,
    0xc0, 0xeb, 0xa8, 0x37, 0x3b, 0x5f, 0xbc, 0x8c, 0x4e, 0x96, 0x25, 0x5c,
    0x1e, 0xb1, 0x35, 0x33, 0x2f, 0xfd, 0x55, 0x32, 0x9b, 0xd3, 0xdd, 0x1d,
    0xe1, 0x8a, 0x4b, 0x1f, 0xf9, 0xd4, 0x0a, 0xa9, 0xf7, 0xdc, 0xae, 0x75,
    0x85, 0x32, 0x66, 0xed, 0x53};

// kExampleECKeyDER is a sample EC private key encoded as an ECPrivateKey
// structure.
static const uint8_t kExampleECKeyDER[] = {
    0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0x07, 0x0f, 0x08, 0x72, 0x7a,
    0xd4, 0xa0, 0x4a, 0x9c, 0xdd, 0x59, 0xc9, 0x4d, 0x89, 0x68, 0x77, 0x08,
    0xb5, 0x6f, 0xc9, 0x5d, 0x30, 0x77, 0x0e, 0xe8, 0xd1, 0xc9, 0xce, 0x0a,
    0x8b, 0xb4, 0x6a, 0xa0, 0x0a, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x03, 0x01, 0x07, 0xa1, 0x44, 0x03, 0x42, 0x00, 0x04, 0xe6, 0x2b, 0x69,
    0xe2, 0xbf, 0x65, 0x9f, 0x97, 0xbe, 0x2f, 0x1e, 0x0d, 0x94, 0x8a, 0x4c,
    0xd5, 0x97, 0x6b, 0xb7, 0xa9, 0x1e, 0x0d, 0x46, 0xfb, 0xdd, 0xa9, 0xa9,
    0x1e, 0x9d, 0xdc, 0xba, 0x5a, 0x01, 0xe7, 0xd6, 0x97, 0xa8, 0x0a, 0x18,
    0xf9, 0xc3, 0xc4, 0xa3, 0x1e, 0x56, 0xe2, 0x7c, 0x83, 0x48, 0xdb, 0x16,
    0x1a, 0x1c, 0xf5, 0x1d, 0x7e, 0xf1, 0x94, 0x2d, 0x4b, 0xcf, 0x72, 0x22,
    0xc1,
};

// kExampleECKeyPKCS8 is a sample EC private key encoded as a PKCS#8
// PrivateKeyInfo.
static const uint8_t kExampleECKeyPKCS8[] = {
    0x30, 0x81, 0x87, 0x02, 0x01, 0x00, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x03, 0x01, 0x07, 0x04, 0x6d, 0x30, 0x6b, 0x02, 0x01, 0x01, 0x04, 0x20,
    0x43, 0x09, 0xc0, 0x67, 0x75, 0x21, 0x47, 0x9d, 0xa8, 0xfa, 0x16, 0xdf,
    0x15, 0x73, 0x61, 0x34, 0x68, 0x6f, 0xe3, 0x8e, 0x47, 0x91, 0x95, 0xab,
    0x79, 0x4a, 0x72, 0x14, 0xcb, 0xe2, 0x49, 0x4f, 0xa1, 0x44, 0x03, 0x42,
    0x00, 0x04, 0xde, 0x09, 0x08, 0x07, 0x03, 0x2e, 0x8f, 0x37, 0x9a, 0xd5,
    0xad, 0xe5, 0xc6, 0x9d, 0xd4, 0x63, 0xc7, 0x4a, 0xe7, 0x20, 0xcb, 0x90,
    0xa0, 0x1f, 0x18, 0x18, 0x72, 0xb5, 0x21, 0x88, 0x38, 0xc0, 0xdb, 0xba,
    0xf6, 0x99, 0xd8, 0xa5, 0x3b, 0x83, 0xe9, 0xe3, 0xd5, 0x61, 0x99, 0x73,
    0x42, 0xc6, 0x6c, 0xe8, 0x0a, 0x95, 0x40, 0x41, 0x3b, 0x0d, 0x10, 0xa7,
    0x4a, 0x93, 0xdb, 0x5a, 0xe7, 0xec,
};

// kExampleECKeySpecifiedCurvePKCS8 is a sample EC private key encoded as a
// PKCS#8 PrivateKeyInfo with P-256's parameters spelled out rather than using
// the curve OID.
static const uint8_t kExampleECKeySpecifiedCurvePKCS8[] = {
    0x30, 0x82, 0x01, 0x79, 0x02, 0x01, 0x00, 0x30, 0x82, 0x01, 0x03, 0x06,
    0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x02, 0x01, 0x30, 0x81, 0xf7, 0x02,
    0x01, 0x01, 0x30, 0x2c, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x01,
    0x01, 0x02, 0x21, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
    0x30, 0x5b, 0x04, 0x20, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfc,
    0x04, 0x20, 0x5a, 0xc6, 0x35, 0xd8, 0xaa, 0x3a, 0x93, 0xe7, 0xb3, 0xeb,
    0xbd, 0x55, 0x76, 0x98, 0x86, 0xbc, 0x65, 0x1d, 0x06, 0xb0, 0xcc, 0x53,
    0xb0, 0xf6, 0x3b, 0xce, 0x3c, 0x3e, 0x27, 0xd2, 0x60, 0x4b, 0x03, 0x15,
    0x00, 0xc4, 0x9d, 0x36, 0x08, 0x86, 0xe7, 0x04, 0x93, 0x6a, 0x66, 0x78,
    0xe1, 0x13, 0x9d, 0x26, 0xb7, 0x81, 0x9f, 0x7e, 0x90, 0x04, 0x41, 0x04,
    0x6b, 0x17, 0xd1, 0xf2, 0xe1, 0x2c, 0x42, 0x47, 0xf8, 0xbc, 0xe6, 0xe5,
    0x63, 0xa4, 0x40, 0xf2, 0x77, 0x03, 0x7d, 0x81, 0x2d, 0xeb, 0x33, 0xa0,
    0xf4, 0xa1, 0x39, 0x45, 0xd8, 0x98, 0xc2, 0x96, 0x4f, 0xe3, 0x42, 0xe2,
    0xfe, 0x1a, 0x7f, 0x9b, 0x8e, 0xe7, 0xeb, 0x4a, 0x7c, 0x0f, 0x9e, 0x16,
    0x2b, 0xce, 0x33, 0x57, 0x6b, 0x31, 0x5e, 0xce, 0xcb, 0xb6, 0x40, 0x68,
    0x37, 0xbf, 0x51, 0xf5, 0x02, 0x21, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00,
    0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xbc,
    0xe6, 0xfa, 0xad, 0xa7, 0x17, 0x9e, 0x84, 0xf3, 0xb9, 0xca, 0xc2, 0xfc,
    0x63, 0x25, 0x51, 0x02, 0x01, 0x01, 0x04, 0x6d, 0x30, 0x6b, 0x02, 0x01,
    0x01, 0x04, 0x20, 0x43, 0x09, 0xc0, 0x67, 0x75, 0x21, 0x47, 0x9d, 0xa8,
    0xfa, 0x16, 0xdf, 0x15, 0x73, 0x61, 0x34, 0x68, 0x6f, 0xe3, 0x8e, 0x47,
    0x91, 0x95, 0xab, 0x79, 0x4a, 0x72, 0x14, 0xcb, 0xe2, 0x49, 0x4f, 0xa1,
    0x44, 0x03, 0x42, 0x00, 0x04, 0xde, 0x09, 0x08, 0x07, 0x03, 0x2e, 0x8f,
    0x37, 0x9a, 0xd5, 0xad, 0xe5, 0xc6, 0x9d, 0xd4, 0x63, 0xc7, 0x4a, 0xe7,
    0x20, 0xcb, 0x90, 0xa0, 0x1f, 0x18, 0x18, 0x72, 0xb5, 0x21, 0x88, 0x38,
    0xc0, 0xdb, 0xba, 0xf6, 0x99, 0xd8, 0xa5, 0x3b, 0x83, 0xe9, 0xe3, 0xd5,
    0x61, 0x99, 0x73, 0x42, 0xc6, 0x6c, 0xe8, 0x0a, 0x95, 0x40, 0x41, 0x3b,
    0x0d, 0x10, 0xa7, 0x4a, 0x93, 0xdb, 0x5a, 0xe7, 0xec,
};

// kExampleBadECKeyDER is a sample EC private key encoded as an ECPrivateKey
// structure. The private key is equal to the order and will fail to import.
static const uint8_t kExampleBadECKeyDER[] = {
    0x30, 0x66, 0x02, 0x01, 0x00, 0x30, 0x13, 0x06, 0x07, 0x2A, 0x86, 0x48,
    0xCE, 0x3D, 0x02, 0x01, 0x06, 0x08, 0x2A, 0x86, 0x48, 0xCE, 0x3D, 0x03,
    0x01, 0x07, 0x04, 0x4C, 0x30, 0x4A, 0x02, 0x01, 0x01, 0x04, 0x20, 0xFF,
    0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xBC, 0xE6, 0xFA, 0xAD, 0xA7, 0x17, 0x9E, 0x84, 0xF3,
    0xB9, 0xCA, 0xC2, 0xFC, 0x63, 0x25, 0x51, 0xA1, 0x23, 0x03, 0x21, 0x00,
    0x00, 0xFF, 0xFF, 0xFF, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF,
    0xFF, 0xFF, 0xFF, 0xFF, 0xBC, 0xE6, 0xFA, 0xAD, 0xA7, 0x17, 0x9E, 0x84,
    0xF3, 0xB9, 0xCA, 0xC2, 0xFC, 0x63, 0x25, 0x51
};

// kExampleBadECKeyDER2 is a sample EC private key encoded as an ECPrivateKey
// structure, but with the curve OID swapped out for 1.1.1.1.1.1.1.1.1. It is
// then concatenated with an ECPrivateKey wrapped in a PrivateKeyInfo,
// optional public key omitted, and with the private key chopped off.
static const uint8_t kExampleBadECKeyDER2[] = {
    0x30, 0x77, 0x02, 0x01, 0x01, 0x04, 0x20, 0x07, 0x0f, 0x08, 0x72, 0x7a,
    0xd4, 0xa0, 0x4a, 0x9c, 0xdd, 0x59, 0xc9, 0x4d, 0x89, 0x68, 0x77, 0x08,
    0xb5, 0x6f, 0xc9, 0x5d, 0x30, 0x77, 0x0e, 0xe8, 0xd1, 0xc9, 0xce, 0x0a,
    0x8b, 0xb4, 0x6a, 0xa0, 0x0a, 0x06, 0x08, 0x29, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0xa1, 0x44, 0x03, 0x42, 0x00, 0x04, 0xe6, 0x2b, 0x69,
    0xe2, 0xbf, 0x65, 0x9f, 0x97, 0xbe, 0x2f, 0x1e, 0x0d, 0x94, 0x8a, 0x4c,
    0xd5, 0x97, 0x6b, 0xb7, 0xa9, 0x1e, 0x0d, 0x46, 0xfb, 0xdd, 0xa9, 0xa9,
    0x1e, 0x9d, 0xdc, 0xba, 0x5a, 0x01, 0xe7, 0xd6, 0x97, 0xa8, 0x0a, 0x18,
    0xf9, 0xc3, 0xc4, 0xa3, 0x1e, 0x56, 0xe2, 0x7c, 0x83, 0x48, 0xdb, 0x16,
    0x1a, 0x1c, 0xf5, 0x1d, 0x7e, 0xf1, 0x94, 0x2d, 0x4b, 0xcf, 0x72, 0x22,
    0xc1, 0x30, 0x41, 0x02, 0x01, 0x00, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86,
    0x48, 0xce, 0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
    0x03, 0x01, 0x07, 0x04, 0x27, 0x30, 0x25, 0x02, 0x01, 0x01, 0x04, 0x20,
    0x07,
};

// kInvalidPrivateKey is an invalid private key. See
// https://rt.openssl.org/Ticket/Display.html?id=4131.
static const uint8_t kInvalidPrivateKey[] = {
    0x30, 0x39, 0x02, 0x01, 0x02, 0x30, 0x09, 0x06, 0x01, 0x38, 0x08,
    0x04, 0x69, 0x30, 0x30, 0x80, 0x30, 0x19, 0x01, 0x02, 0x9f, 0xf8,
    0x8b, 0x29, 0x80, 0x30, 0xb0, 0x1b, 0x06, 0x09, 0x22, 0xbe, 0x08,
    0x04, 0xe9, 0x30, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x3a, 0x01, 0x80,
    0x09, 0x30, 0x80, 0x06, 0x01, 0x02, 0x30, 0x80, 0x30, 0x01, 0x3b,
    0x02, 0x00, 0x00, 0x04, 0x20, 0x30, 0x82, 0x04, 0xe9, 0x30, 0xc3,
    0xe8, 0x30, 0x01, 0x05, 0x30, 0x80, 0x30, 0x01, 0x3b, 0x01, 0x04,
    0x02, 0x02, 0xff, 0x00, 0x30, 0x29, 0x02, 0x11, 0x03, 0x29, 0x29,
    0x02, 0x00, 0x99, 0x30, 0x80, 0x06, 0x21, 0x02, 0x24, 0x04, 0xe8,
    0x30, 0x01, 0x01, 0x04, 0x30, 0x80, 0x1b, 0x06, 0x09, 0x2a, 0x86,
    0x48, 0x30, 0x01, 0xaa, 0x02, 0x86, 0xc0, 0x30, 0xdf, 0xe9, 0x80,
};

static bssl::UniquePtr<EVP_PKEY> LoadExampleRSAKey() {
  bssl::UniquePtr<RSA> rsa(RSA_private_key_from_bytes(kExampleRSAKeyDER,
                                           sizeof(kExampleRSAKeyDER)));
  if (!rsa) {
    return nullptr;
  }
  bssl::UniquePtr<EVP_PKEY> pkey(EVP_PKEY_new());
  if (!pkey || !EVP_PKEY_set1_RSA(pkey.get(), rsa.get())) {
    return nullptr;
  }
  return pkey;
}

TEST(EVPExtraTest, DigestSignInit) {
  bssl::UniquePtr<EVP_PKEY> pkey = LoadExampleRSAKey();
  ASSERT_TRUE(pkey);
  bssl::ScopedEVP_MD_CTX md_ctx;
  ASSERT_TRUE(
      EVP_DigestSignInit(md_ctx.get(), NULL, EVP_sha256(), NULL, pkey.get()));
  ASSERT_TRUE(EVP_DigestSignUpdate(md_ctx.get(), kMsg, sizeof(kMsg)));

  // Determine the size of the signature.
  size_t sig_len = 0;
  ASSERT_TRUE(EVP_DigestSignFinal(md_ctx.get(), NULL, &sig_len));

  // Sanity check for testing.
  EXPECT_EQ(static_cast<size_t>(EVP_PKEY_size(pkey.get())), sig_len);

  std::vector<uint8_t> sig;
  sig.resize(sig_len);
  ASSERT_TRUE(EVP_DigestSignFinal(md_ctx.get(), sig.data(), &sig_len));
  sig.resize(sig_len);

  // Ensure that the signature round-trips.
  md_ctx.Reset();
  ASSERT_TRUE(
      EVP_DigestVerifyInit(md_ctx.get(), NULL, EVP_sha256(), NULL, pkey.get()));
  ASSERT_TRUE(EVP_DigestVerifyUpdate(md_ctx.get(), kMsg, sizeof(kMsg)));
  ASSERT_TRUE(EVP_DigestVerifyFinal(md_ctx.get(), sig.data(), sig_len));
}

TEST(EVPExtraTest, DigestVerifyInit) {
  bssl::UniquePtr<EVP_PKEY> pkey = LoadExampleRSAKey();
  bssl::ScopedEVP_MD_CTX md_ctx;
  ASSERT_TRUE(pkey);
  ASSERT_TRUE(
      EVP_DigestVerifyInit(md_ctx.get(), NULL, EVP_sha256(), NULL, pkey.get()));
  ASSERT_TRUE(EVP_DigestVerifyUpdate(md_ctx.get(), kMsg, sizeof(kMsg)));
  ASSERT_TRUE(
      EVP_DigestVerifyFinal(md_ctx.get(), kSignature, sizeof(kSignature)));
}

TEST(EVPExtraTest, VerifyRecover) {
  bssl::UniquePtr<EVP_PKEY> pkey = LoadExampleRSAKey();
  ASSERT_TRUE(pkey);
  bssl::UniquePtr<RSA> rsa(EVP_PKEY_get1_RSA(pkey.get()));
  ASSERT_TRUE(rsa);

  const uint8_t kDummyHash[32] = {0};
  uint8_t sig[2048/8];
  unsigned sig_len = sizeof(sig);
  ASSERT_TRUE(RSA_sign(NID_sha256, kDummyHash, sizeof(kDummyHash), sig,
                       &sig_len, rsa.get()));

  size_t out_len;
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new(pkey.get(), nullptr));
  ASSERT_TRUE(EVP_PKEY_verify_recover_init(ctx.get()));
  ASSERT_TRUE(EVP_PKEY_CTX_set_rsa_padding(ctx.get(), RSA_PKCS1_PADDING));
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(ctx.get(), EVP_sha256()));
  ASSERT_TRUE(
      EVP_PKEY_verify_recover(ctx.get(), nullptr, &out_len, sig, sig_len));

  std::vector<uint8_t> recovered;
  recovered.resize(out_len);

  ASSERT_TRUE(EVP_PKEY_verify_recover(ctx.get(), recovered.data(), &out_len,
                                      sig, sig_len));
  EXPECT_EQ(Bytes(kDummyHash), Bytes(recovered.data(), out_len));

  out_len = recovered.size();
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(ctx.get(), nullptr));
  ASSERT_TRUE(EVP_PKEY_verify_recover(ctx.get(), recovered.data(), &out_len,
                                      sig, sig_len));

  // The size of a SHA-256 hash plus PKCS#1 v1.5 ASN.1 stuff happens to be 51
  // bytes.
  EXPECT_EQ(51u, out_len);
}

static void TestValidPrivateKey(const uint8_t *input, size_t input_len,
                                int expected_id) {
  const uint8_t *p = input;
  bssl::UniquePtr<EVP_PKEY> pkey(d2i_AutoPrivateKey(NULL, &p, input_len));
  ASSERT_TRUE(pkey);
  EXPECT_EQ(input + input_len, p);
  EXPECT_EQ(expected_id, EVP_PKEY_id(pkey.get()));
}

TEST(EVPExtraTest, d2i_AutoPrivateKey) {
  TestValidPrivateKey(kExampleRSAKeyDER, sizeof(kExampleRSAKeyDER),
                      EVP_PKEY_RSA);
  TestValidPrivateKey(kExampleRSAKeyPKCS8, sizeof(kExampleRSAKeyPKCS8),
                      EVP_PKEY_RSA);
  TestValidPrivateKey(kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8),
                      EVP_PKEY_RSA_PSS);
  TestValidPrivateKey(kExampleRSAPSSKeyNoPSSParams,
                      sizeof(kExampleRSAPSSKeyNoPSSParams),
                      EVP_PKEY_RSA_PSS);
  TestValidPrivateKey(kExampleECKeyDER, sizeof(kExampleECKeyDER), EVP_PKEY_EC);
  TestValidPrivateKey(kExampleECKeyPKCS8, sizeof(kExampleECKeyPKCS8),
                      EVP_PKEY_EC);
  TestValidPrivateKey(kExampleECKeySpecifiedCurvePKCS8,
                      sizeof(kExampleECKeySpecifiedCurvePKCS8), EVP_PKEY_EC);
  TestValidPrivateKey(kExampleDSAKeyDER, sizeof(kExampleDSAKeyDER),
                      EVP_PKEY_DSA);

  const uint8_t *p = kInvalidPrivateKey;
  bssl::UniquePtr<EVP_PKEY> pkey(
      d2i_AutoPrivateKey(NULL, &p, sizeof(kInvalidPrivateKey)));
  EXPECT_FALSE(pkey) << "Parsed invalid private key";
  ERR_clear_error();
}

// Tests loading a bad key in PKCS8 format.
TEST(EVPExtraTest, BadECKey) {
  const uint8_t *derp = kExampleBadECKeyDER;
  bssl::UniquePtr<PKCS8_PRIV_KEY_INFO> p8inf(
      d2i_PKCS8_PRIV_KEY_INFO(NULL, &derp, sizeof(kExampleBadECKeyDER)));
  ASSERT_TRUE(p8inf);
  EXPECT_EQ(kExampleBadECKeyDER + sizeof(kExampleBadECKeyDER), derp);

  bssl::UniquePtr<EVP_PKEY> pkey(EVP_PKCS82PKEY(p8inf.get()));
  ASSERT_FALSE(pkey) << "Imported invalid EC key";
  ERR_clear_error();
}

// Tests |EVP_marshal_public_key| on an empty key.
TEST(EVPExtraTest, MarshalEmptyPublicKey) {
  bssl::UniquePtr<EVP_PKEY> empty(EVP_PKEY_new());
  ASSERT_TRUE(empty);

  bssl::ScopedCBB cbb;
  EXPECT_FALSE(EVP_marshal_public_key(cbb.get(), empty.get()))
      << "Marshalled empty public key.";
  EXPECT_EQ(EVP_R_UNSUPPORTED_ALGORITHM, ERR_GET_REASON(ERR_peek_last_error()));
}

static bssl::UniquePtr<EVP_PKEY> ParsePrivateKey(int type, const uint8_t *in,
                                                 size_t len) {
  const uint8_t *ptr = in;
  bssl::UniquePtr<EVP_PKEY> pkey(d2i_PrivateKey(type, nullptr, &ptr, len));
  if (!pkey) {
    return nullptr;
  }

  EXPECT_EQ(in + len, ptr);
  return pkey;
}

TEST(EVPExtraTest, d2i_PrivateKey) {
  EXPECT_TRUE(ParsePrivateKey(EVP_PKEY_RSA, kExampleRSAKeyDER,
                              sizeof(kExampleRSAKeyDER)));
  EXPECT_TRUE(ParsePrivateKey(EVP_PKEY_DSA, kExampleDSAKeyDER,
                              sizeof(kExampleDSAKeyDER)));
  EXPECT_TRUE(ParsePrivateKey(EVP_PKEY_RSA, kExampleRSAKeyPKCS8,
                              sizeof(kExampleRSAKeyPKCS8)));
  EXPECT_TRUE(ParsePrivateKey(EVP_PKEY_RSA_PSS, kExampleRSAPSSKeyPKCS8,
                              sizeof(kExampleRSAPSSKeyPKCS8)));
  EXPECT_TRUE(
      ParsePrivateKey(EVP_PKEY_EC, kExampleECKeyDER, sizeof(kExampleECKeyDER)));

  EXPECT_FALSE(ParsePrivateKey(EVP_PKEY_EC, kExampleBadECKeyDER,
                               sizeof(kExampleBadECKeyDER)));
  ERR_clear_error();

  // Copy the input into a |malloc|'d vector to flag memory errors.
  std::vector<uint8_t> copy(
      kExampleBadECKeyDER2,
      kExampleBadECKeyDER2 + sizeof(kExampleBadECKeyDER2));
  EXPECT_FALSE(ParsePrivateKey(EVP_PKEY_EC, copy.data(), copy.size()));
  ERR_clear_error();

  // Test that an RSA key may not be imported as an EC key.
  EXPECT_FALSE(ParsePrivateKey(EVP_PKEY_EC, kExampleRSAKeyPKCS8,
                               sizeof(kExampleRSAKeyPKCS8)));
  EXPECT_FALSE(ParsePrivateKey(EVP_PKEY_EC, kExampleRSAPSSKeyPKCS8,
                               sizeof(kExampleRSAPSSKeyPKCS8)));
  ERR_clear_error();
}

TEST(EVPExtraTest, Ed25519) {
  static const uint8_t kPublicKey[32] = {
      0xd7, 0x5a, 0x98, 0x01, 0x82, 0xb1, 0x0a, 0xb7, 0xd5, 0x4b, 0xfe,
      0xd3, 0xc9, 0x64, 0x07, 0x3a, 0x0e, 0xe1, 0x72, 0xf3, 0xda, 0xa6,
      0x23, 0x25, 0xaf, 0x02, 0x1a, 0x68, 0xf7, 0x07, 0x51, 0x1a,
  };

  static const uint8_t kPublicKeySPKI[] = {
      0x30, 0x2a, 0x30, 0x05, 0x06, 0x03, 0x2b, 0x65, 0x70, 0x03, 0x21,
      0x00, 0xd7, 0x5a, 0x98, 0x01, 0x82, 0xb1, 0x0a, 0xb7, 0xd5, 0x4b,
      0xfe, 0xd3, 0xc9, 0x64, 0x07, 0x3a, 0x0e, 0xe1, 0x72, 0xf3, 0xda,
      0xa6, 0x23, 0x25, 0xaf, 0x02, 0x1a, 0x68, 0xf7, 0x07, 0x51, 0x1a,
  };

  static const uint8_t kPrivateKeySeed[32] = {
      0x9d, 0x61, 0xb1, 0x9d, 0xef, 0xfd, 0x5a, 0x60, 0xba, 0x84, 0x4a,
      0xf4, 0x92, 0xec, 0x2c, 0xc4, 0x44, 0x49, 0xc5, 0x69, 0x7b, 0x32,
      0x69, 0x19, 0x70, 0x3b, 0xac, 0x03, 0x1c, 0xae, 0x7f, 0x60,
  };

  static const uint8_t kPrivateKeyPKCS8[] = {
      0x30, 0x2e, 0x02, 0x01, 0x00, 0x30, 0x05, 0x06, 0x03, 0x2b, 0x65, 0x70,
      0x04, 0x22, 0x04, 0x20, 0x9d, 0x61, 0xb1, 0x9d, 0xef, 0xfd, 0x5a, 0x60,
      0xba, 0x84, 0x4a, 0xf4, 0x92, 0xec, 0x2c, 0xc4, 0x44, 0x49, 0xc5, 0x69,
      0x7b, 0x32, 0x69, 0x19, 0x70, 0x3b, 0xac, 0x03, 0x1c, 0xae, 0x7f, 0x60,
  };

  // Create a public key.
  bssl::UniquePtr<EVP_PKEY> pubkey(EVP_PKEY_new_raw_public_key(
      EVP_PKEY_ED25519, nullptr, kPublicKey, sizeof(kPublicKey)));
  ASSERT_TRUE(pubkey);
  EXPECT_EQ(EVP_PKEY_ED25519, EVP_PKEY_id(pubkey.get()));

  // The public key must be extractable.
  uint8_t buf[32];
  size_t len;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pubkey.get(), nullptr, &len));
  EXPECT_EQ(len, 32u);
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pubkey.get(), buf, &len));
  EXPECT_EQ(Bytes(buf, len), Bytes(kPublicKey));
  // Passing too large of a buffer is okay. The function will still only read
  // 32 bytes.
  len = 64;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(pubkey.get(), buf, &len));
  EXPECT_EQ(Bytes(buf, len), Bytes(kPublicKey));
  // Passing too small of a buffer is noticed.
  len = 31;
  EXPECT_FALSE(EVP_PKEY_get_raw_public_key(pubkey.get(), buf, &len));
  uint32_t err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  ERR_clear_error();

  // There is no private key.
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(pubkey.get(), nullptr, &len));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_NOT_A_PRIVATE_KEY, ERR_GET_REASON(err));
  ERR_clear_error();

  // The public key must encode properly.
  bssl::ScopedCBB cbb;
  uint8_t *der;
  size_t der_len;
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), pubkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  bssl::UniquePtr<uint8_t> free_der(der);
  EXPECT_EQ(Bytes(kPublicKeySPKI), Bytes(der, der_len));

  // The public key must gracefully fail to encode as a private key.
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  EXPECT_FALSE(EVP_marshal_private_key(cbb.get(), pubkey.get()));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_NOT_A_PRIVATE_KEY, ERR_GET_REASON(err));
  ERR_clear_error();
  cbb.Reset();

  // Create a private key.
  bssl::UniquePtr<EVP_PKEY> privkey(EVP_PKEY_new_raw_private_key(
      EVP_PKEY_ED25519, NULL, kPrivateKeySeed, sizeof(kPrivateKeySeed)));
  ASSERT_TRUE(privkey);
  EXPECT_EQ(EVP_PKEY_ED25519, EVP_PKEY_id(privkey.get()));

  // The private key must be extractable.
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(privkey.get(), nullptr, &len));
  EXPECT_EQ(len, 32u);
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(privkey.get(), buf, &len));
  EXPECT_EQ(Bytes(buf, len), Bytes(kPrivateKeySeed));
  // Passing too large of a buffer is okay. The function will still only read
  // 32 bytes.
  len = 64;
  ASSERT_TRUE(EVP_PKEY_get_raw_private_key(privkey.get(), buf, &len));
  EXPECT_EQ(Bytes(buf, len), Bytes(kPrivateKeySeed));
  // Passing too small of a buffer is noticed.
  len = 31;
  EXPECT_FALSE(EVP_PKEY_get_raw_private_key(privkey.get(), buf, &len));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  ERR_clear_error();
  // The public key must be extractable.
  len = 32;
  ASSERT_TRUE(EVP_PKEY_get_raw_public_key(privkey.get(), buf, &len));
  EXPECT_EQ(Bytes(buf, len), Bytes(kPublicKey));

  // The public key must encode from the private key.
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_public_key(cbb.get(), privkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  free_der.reset(der);
  EXPECT_EQ(Bytes(kPublicKeySPKI), Bytes(der, der_len));

  // The private key must encode properly.
  ASSERT_TRUE(CBB_init(cbb.get(), 0));
  ASSERT_TRUE(EVP_marshal_private_key(cbb.get(), privkey.get()));
  ASSERT_TRUE(CBB_finish(cbb.get(), &der, &der_len));
  free_der.reset(der);
  EXPECT_EQ(Bytes(kPrivateKeyPKCS8), Bytes(der, der_len));

  // Test EVP_PKEY_cmp.
  EXPECT_EQ(1, EVP_PKEY_cmp(pubkey.get(), privkey.get()));

  static const uint8_t kZeros[32] = {0};
  bssl::UniquePtr<EVP_PKEY> pubkey2(EVP_PKEY_new_raw_public_key(
      EVP_PKEY_ED25519, nullptr, kZeros, sizeof(kZeros)));
  ASSERT_TRUE(pubkey2);
  EXPECT_EQ(0, EVP_PKEY_cmp(pubkey.get(), pubkey2.get()));
  EXPECT_EQ(0, EVP_PKEY_cmp(privkey.get(), pubkey2.get()));

  // Ed25519 may not be used streaming.
  bssl::ScopedEVP_MD_CTX ctx;
  ASSERT_TRUE(
      EVP_DigestSignInit(ctx.get(), nullptr, nullptr, nullptr, privkey.get()));
  EXPECT_FALSE(EVP_DigestSignUpdate(ctx.get(), nullptr, 0));
  EXPECT_FALSE(EVP_DigestSignFinal(ctx.get(), nullptr, &len));
  ERR_clear_error();

  ctx.Reset();
  ASSERT_TRUE(EVP_DigestVerifyInit(ctx.get(), nullptr, nullptr, nullptr,
                                   privkey.get()));
  EXPECT_FALSE(EVP_DigestVerifyUpdate(ctx.get(), nullptr, 0));
  EXPECT_FALSE(EVP_DigestVerifyFinal(ctx.get(), nullptr, 0));
  ERR_clear_error();

  // The buffer length to |EVP_DigestSign| is an input/output parameter and
  // should be checked before signing.
  ctx.Reset();
  ASSERT_TRUE(
      EVP_DigestSignInit(ctx.get(), nullptr, nullptr, nullptr, privkey.get()));
  len = 31;
  EXPECT_FALSE(EVP_DigestSign(ctx.get(), buf, &len, nullptr /* msg */, 0));
  err = ERR_get_error();
  EXPECT_EQ(ERR_LIB_EVP, ERR_GET_LIB(err));
  EXPECT_EQ(EVP_R_BUFFER_TOO_SMALL, ERR_GET_REASON(err));
  ERR_clear_error();
}

static void ExpectECGroupOnly(const EVP_PKEY *pkey, int nid) {
  EC_KEY *ec = EVP_PKEY_get0_EC_KEY(pkey);
  ASSERT_TRUE(ec);
  const EC_GROUP *group = EC_KEY_get0_group(ec);
  ASSERT_TRUE(group);
  EXPECT_EQ(nid, EC_GROUP_get_curve_name(group));
  EXPECT_FALSE(EC_KEY_get0_public_key(ec));
  EXPECT_FALSE(EC_KEY_get0_private_key(ec));
}

static void ExpectECGroupAndKey(const EVP_PKEY *pkey, int nid) {
  EC_KEY *ec = EVP_PKEY_get0_EC_KEY(pkey);
  ASSERT_TRUE(ec);
  const EC_GROUP *group = EC_KEY_get0_group(ec);
  ASSERT_TRUE(group);
  EXPECT_EQ(nid, EC_GROUP_get_curve_name(group));
  EXPECT_TRUE(EC_KEY_get0_public_key(ec));
  EXPECT_TRUE(EC_KEY_get0_private_key(ec));
}

TEST(EVPExtraTest, ECKeygen) {
  // |EVP_PKEY_paramgen| may be used as an extremely roundabout way to get an
  // |EC_GROUP|.
  bssl::UniquePtr<EVP_PKEY_CTX> ctx(EVP_PKEY_CTX_new_id(EVP_PKEY_EC, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_paramgen_init(ctx.get()));
  ASSERT_TRUE(
      EVP_PKEY_CTX_set_ec_paramgen_curve_nid(ctx.get(), NID_X9_62_prime256v1));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_paramgen(ctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);
  raw = nullptr;
  ExpectECGroupOnly(pkey.get(), NID_X9_62_prime256v1);

  // That resulting |EVP_PKEY| may be used as a template for key generation.
  ctx.reset(EVP_PKEY_CTX_new(pkey.get(), nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  pkey.reset(raw);
  raw = nullptr;
  ExpectECGroupAndKey(pkey.get(), NID_X9_62_prime256v1);

  // |EVP_PKEY_paramgen| may also be skipped.
  ctx.reset(EVP_PKEY_CTX_new_id(EVP_PKEY_EC, nullptr));
  ASSERT_TRUE(ctx);
  ASSERT_TRUE(EVP_PKEY_keygen_init(ctx.get()));
  ASSERT_TRUE(
      EVP_PKEY_CTX_set_ec_paramgen_curve_nid(ctx.get(), NID_X9_62_prime256v1));
  raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(ctx.get(), &raw));
  pkey.reset(raw);
  raw = nullptr;
  ExpectECGroupAndKey(pkey.get(), NID_X9_62_prime256v1);
}

// Test that |EVP_PKEY_keygen| works for Ed25519.
TEST(EVPExtraTest, Ed25519Keygen) {
  bssl::UniquePtr<EVP_PKEY_CTX> pctx(
      EVP_PKEY_CTX_new_id(EVP_PKEY_ED25519, nullptr));
  ASSERT_TRUE(pctx);
  ASSERT_TRUE(EVP_PKEY_keygen_init(pctx.get()));
  EVP_PKEY *raw = nullptr;
  ASSERT_TRUE(EVP_PKEY_keygen(pctx.get(), &raw));
  bssl::UniquePtr<EVP_PKEY> pkey(raw);

  // Round-trip a signature to sanity-check the key is good.
  bssl::ScopedEVP_MD_CTX ctx;
  ASSERT_TRUE(
      EVP_DigestSignInit(ctx.get(), nullptr, nullptr, nullptr, pkey.get()));
  uint8_t sig[64];
  size_t len = sizeof(sig);
  ASSERT_TRUE(EVP_DigestSign(ctx.get(), sig, &len,
                             reinterpret_cast<const uint8_t *>("hello"), 5));

  ctx.Reset();
  ASSERT_TRUE(
      EVP_DigestVerifyInit(ctx.get(), nullptr, nullptr, nullptr, pkey.get()));
  ASSERT_TRUE(EVP_DigestVerify(ctx.get(), sig, len,
                               reinterpret_cast<const uint8_t *>("hello"), 5));
}

// Test that |EVP_DigestSignFinal| and |EVP_DigestSignVerify| work with a
// a special use case of not using the one-shot |EVP_DigestSignInit| or
// |EVP_DigestVerifyInit| to initialize the |EVP_PKEY_CTX| context. The context
// data can be manually constructed using other context setting functions.
// |EVP_MD_CTX_set_pkey_ctx| was added to support this use case.
TEST(EVPExtraTest, PKEY_CTX_manual) {
  // Test the behavior with RSA Keys.
  bssl::ScopedEVP_MD_CTX ctx;
  ASSERT_TRUE(EVP_DigestInit(ctx.get(), EVP_sha256()));
  ASSERT_TRUE(EVP_DigestUpdate(ctx.get(), kMsg, sizeof(kMsg)));

  bssl::UniquePtr<EVP_PKEY> pkey = LoadExampleRSAKey();
  ASSERT_TRUE(pkey);
  bssl::UniquePtr<EVP_PKEY_CTX> pctx(EVP_PKEY_CTX_new(pkey.get(), nullptr));
  ASSERT_TRUE(pctx);

  // Manual construction for signing.
  ASSERT_TRUE(EVP_PKEY_sign_init(pctx.get()));
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(pctx.get(), EVP_sha256()));
  EVP_MD_CTX_set_pkey_ctx(ctx.get(), pctx.get());
  // Determine the size of the signature.
  size_t sig_len = 0;
  ASSERT_TRUE(EVP_DigestSignFinal(ctx.get(), nullptr, &sig_len));

  std::vector<uint8_t> sig;
  sig.resize(sig_len);
  ASSERT_TRUE(EVP_DigestSignFinal(ctx.get(), sig.data(), &sig_len));
  sig.resize(sig_len);

  // Manual construction for verification.
  ASSERT_TRUE(EVP_PKEY_verify_init(pctx.get()));
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(pctx.get(), EVP_sha256()));
  EVP_MD_CTX_set_pkey_ctx(ctx.get(), pctx.get());

  ASSERT_TRUE(EVP_DigestVerifyFinal(ctx.get(), sig.data(), sig_len));

  // Test the behavior with EC Keys.
  ctx.Reset();
  ASSERT_TRUE(EVP_DigestInit(ctx.get(), EVP_sha256()));
  ASSERT_TRUE(EVP_DigestUpdate(ctx.get(), kMsg, sizeof(kMsg)));

  // Generate generic |EC_KEY|.
  pkey.reset(EVP_PKEY_new());
  ASSERT_TRUE(pkey);
  bssl::UniquePtr<EC_GROUP> group(EC_GROUP_new_by_curve_name(NID_X9_62_prime256v1));
  bssl::UniquePtr<EC_KEY> eckey(EC_KEY_new());
  ASSERT_TRUE(eckey);
  ASSERT_TRUE(EC_KEY_set_group(eckey.get(), group.get()));
  EC_KEY_generate_key(eckey.get());
  ASSERT_TRUE(EVP_PKEY_set1_EC_KEY(pkey.get(), eckey.get()));
  pctx.reset(EVP_PKEY_CTX_new(pkey.get(), nullptr));

  // Manual construction for signing.
  ASSERT_TRUE(EVP_PKEY_sign_init(pctx.get()));
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(pctx.get(), EVP_sha256()));
  EVP_MD_CTX_set_pkey_ctx(ctx.get(), pctx.get());

  // Determine the size of the signature.
  sig_len = 0;
  ASSERT_TRUE(EVP_DigestSignFinal(ctx.get(), nullptr, &sig_len));

  sig.resize(sig_len);
  ASSERT_TRUE(EVP_DigestSignFinal(ctx.get(), sig.data(), &sig_len));
  sig.resize(sig_len);

  // Manual construction for verification.
  ASSERT_TRUE(EVP_PKEY_verify_init(pctx.get()));
  ASSERT_TRUE(EVP_PKEY_CTX_set_signature_md(pctx.get(), EVP_sha256()));
  EVP_MD_CTX_set_pkey_ctx(ctx.get(), pctx.get());

  ASSERT_TRUE(EVP_DigestVerifyFinal(ctx.get(), sig.data(), sig_len));

  // Test clean up behavior after using |EVP_MD_CTX_set_pkey_ctx|. The
  // externally set |pctx| should not be inadvertently freed.
  ASSERT_TRUE(ctx.get()->md_data);
  ASSERT_TRUE(ctx.get()->pctx);
  ASSERT_TRUE(pctx.get());
  // Clean up here.
  ASSERT_TRUE(EVP_MD_CTX_cleanup(ctx.get()));
  ASSERT_TRUE(ctx.get());
  ASSERT_FALSE(ctx.get()->md_data);
  ASSERT_FALSE(ctx.get()->pctx);
  ASSERT_TRUE(pctx.get());

  // Test assigning a NULL |pctx| with |EVP_MD_CTX_set_pkey_ctx|.
  bssl::ScopedEVP_MD_CTX md_ctx;
  bssl::UniquePtr<EVP_PKEY_CTX> null_pctx(nullptr);
  ASSERT_FALSE(null_pctx.get());
  // Assign a non-NULL pctx first.
  EVP_MD_CTX_set_pkey_ctx(md_ctx.get(), pctx.get());
  ASSERT_TRUE(md_ctx.get()->pctx);
  // Set again with NUll |pctx|.
  EVP_MD_CTX_set_pkey_ctx(md_ctx.get(), null_pctx.get());
  ASSERT_FALSE(md_ctx.get()->pctx);
  ASSERT_FALSE(null_pctx.get());
  // Check that clean up works.
  ASSERT_TRUE(EVP_MD_CTX_cleanup(md_ctx.get()));
  ASSERT_TRUE(ctx.get());
  ASSERT_FALSE(ctx.get()->md_data);
  ASSERT_FALSE(ctx.get()->pctx);
  ASSERT_TRUE(pctx.get());
}

struct RsassaPssParamsMatchTestInput {
  const uint8_t *der;
  size_t der_len;
  const EVP_MD *signature_md;
  const EVP_MD *rsa_mgf1_md;
  int success;
} kRsassaPssParamsMatchTestInputs[] = {
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha1(),
     EVP_sha1(), 0},
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha224(),
     EVP_sha224(), 0},
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha256(),
     EVP_sha256(), 1},
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha384(),
     EVP_sha384(), 0},
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha512(),
     EVP_sha512(), 0},
    {kExampleRSAPSSKeyPKCS8, sizeof(kExampleRSAPSSKeyPKCS8), EVP_sha512_256(),
     EVP_sha512_256(), 0},
    // This test expects success when setting |signature_md| and |rsa_mgf1_md|
    // because |kExampleRSAPSSKeyNoPSSParams| has no pss restriction.
    {kExampleRSAPSSKeyNoPSSParams, sizeof(kExampleRSAPSSKeyNoPSSParams), EVP_sha256(),
     EVP_sha256(), 1},
};

class EVPRsaPssExtraTest
    : public testing::TestWithParam<RsassaPssParamsMatchTestInput> {};

// This test checks params match.
// All parameters in the signature structure algorithm identifier MUST
// match the parameters in the key structure algorithm identifier except
// the saltLength field.
// See 3.3. https://tools.ietf.org/html/rfc4055#section-3.3
TEST_P(EVPRsaPssExtraTest, PssParamsMatch) {
  const auto &param = GetParam();
  const uint8_t *p = param.der;
  const EVP_MD *signature_md = param.signature_md;
  const EVP_MD *rsa_mgf1_md = param.rsa_mgf1_md;
  // Holds ownership of heap-allocated EVP_PKEY.
  bssl::UniquePtr<EVP_PKEY> pkey_up(
      d2i_AutoPrivateKey(nullptr, &p, param.der_len));
  ASSERT_TRUE(pkey_up);
  EXPECT_EQ(param.der + param.der_len, p);
  EXPECT_EQ(EVP_PKEY_RSA_PSS, EVP_PKEY_id(pkey_up.get()));
  // Holds ownership of heap-allocated EVP_PKEY_CTX.
  bssl::UniquePtr<EVP_PKEY_CTX> pkey_ctx_up(
      EVP_PKEY_CTX_new(pkey_up.get(), nullptr));
  ASSERT_TRUE(pkey_ctx_up);
  EVP_PKEY_CTX *pkey_ctx = pkey_ctx_up.get();
  // Init pss params by calling |EVP_PKEY_sign_init|.
  // These pss params are fetched from the key structure.
  ASSERT_TRUE(EVP_PKEY_sign_init(pkey_ctx));
  EXPECT_TRUE(EVP_PKEY_CTX_set_rsa_padding(pkey_ctx, RSA_PKCS1_PSS_PADDING));
  // Compare one way hash func.
  EXPECT_EQ(EVP_PKEY_CTX_set_signature_md(pkey_ctx, signature_md), param.success);
  // Compare one way hash func of mask gen.
  EXPECT_EQ(EVP_PKEY_CTX_set_rsa_mgf1_md(pkey_ctx, rsa_mgf1_md), param.success);
  EXPECT_TRUE(
      EVP_PKEY_CTX_set_rsa_pss_saltlen(pkey_ctx, RSA_PSS_SALTLEN_DIGEST));
  ERR_clear_error();
}

INSTANTIATE_TEST_SUITE_P(All, EVPRsaPssExtraTest,
                         testing::ValuesIn(kRsassaPssParamsMatchTestInputs));

struct BadPssKeyTestInput {
  const uint8_t *der;
  size_t der_len;
} kBadPssKeyTestInputs[] = {
    {badRSAPSSKeyPKCS8_SaltLengthTooLarge, sizeof(badRSAPSSKeyPKCS8_SaltLengthTooLarge)},
};

class EVPRsaPssBadKeyTest : public testing::TestWithParam<BadPssKeyTestInput> {
};

// This test checks pss salt length.
TEST_P(EVPRsaPssBadKeyTest, InvalidSaltLength) {
  const auto &param = GetParam();
  const uint8_t *p = param.der;
  // Holds ownership of heap-allocated EVP_PKEY.
  bssl::UniquePtr<EVP_PKEY> pkey_up(
      d2i_AutoPrivateKey(nullptr, &p, param.der_len));
  ASSERT_TRUE(pkey_up);
  EXPECT_EQ(param.der + param.der_len, p);
  EXPECT_EQ(EVP_PKEY_RSA_PSS, EVP_PKEY_id(pkey_up.get()));
  // Holds ownership of heap-allocated EVP_PKEY_CTX.
  bssl::UniquePtr<EVP_PKEY_CTX> pkey_ctx_up(
      EVP_PKEY_CTX_new(pkey_up.get(), nullptr));
  ASSERT_TRUE(pkey_ctx_up);
  EVP_PKEY_CTX *pkey_ctx = pkey_ctx_up.get();
  // Init pss params by calling |EVP_PKEY_sign_init|.
  // During initialization, the salt length is validated.
  ASSERT_FALSE(EVP_PKEY_sign_init(pkey_ctx));
  ERR_clear_error();
}

INSTANTIATE_TEST_SUITE_P(All, EVPRsaPssBadKeyTest,
                         testing::ValuesIn(kBadPssKeyTestInputs));
