/*
 * MIT License
 *
 * Copyright (c) 2019 Alexei Sintotski
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

import 'dart:convert';
import 'dart:io';

import 'package:json2yaml/json2yaml.dart';
import 'package:json2yaml/src/json2yaml.dart';
import 'package:test/test.dart';
import 'package:yaml/yaml.dart';

import 'test_utils.dart';

void main() {
  group('json2yaml', () {
    testFormatting(
      'Formats version number correctly',
      json: _versionJson,
      yaml: _versionYaml,
      yamlStyle: YamlStyle.pubspecLock,
    );

    testFormatting(
      'Formats direct dev dependency spec correctly',
      json: _directDevDependencySpecJson,
      yaml: _directDevDependencySpecYaml,
      yamlStyle: YamlStyle.pubspecLock,
    );

    test('Preserves formatting of realistic pubspec.lock file', () {
      final originalYaml = File('pubspec.lock').readAsStringSync();
      final dynamic yaml = loadYaml(originalYaml);
      final contentAsJson =
          json.decode(json.encode(yaml)) as Map<String, dynamic>;
      final processedYaml = json2yaml(
        contentAsJson,
        yamlStyle: YamlStyle.pubspecLock,
      );

      final correctedResult = '$_pubspecLockHeader$processedYaml';
      expect(correctedResult, originalYaml);
    });
  });
}

const _pubspecLockHeader = '''
# Generated by pub
# See https://dart.dev/tools/pub/glossary#lockfile
''';

const _versionJson = {'version': '2.3.0'};
const _versionYaml = '''
version: "2.3.0"
''';

const _directDevDependencySpecJson = {'dependency': 'direct dev'};
const _directDevDependencySpecYaml = '''
dependency: "direct dev"
''';
