/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2002-2009, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.jterm;

import java.awt.Event;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;
import javax.swing.KeyStroke;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.text.Keymap;
import javax.swing.text.StyledEditorKit;
import javax.swing.text.TextAction;
import javax.swing.text.Utilities;

/**
 * This is the set of actions needed by the JTerm.
 * all action are made as public static inner classes that will be returned 
 * with getAction() as array.
 */
public class JTermKit extends StyledEditorKit {
  /** Name of the action to evaluate the given string.*/
  public static final String evaluateAction= "Evaluate";
  /** Name of the action to compelte command in the shell. */
  public static final String completeCommandAction= "CompleteCommand";
  /** Name of the action to return to next in history saved command. */
  public static final String historyNextAction= "HistoryNext";
  /** Name of the action to return to previous in history saved command. */
  public static final String historyPrevAction= "HistoryPrev";
  /** Name of action to move caret left till prompt .*/
  public static final String moveLeftAction= "Left";
  /** Name of action to move caret to the end of line. */
  public static final String setToEndOfLineAction= "end-line";
  /** Name of action to move caret to the begin of line. */
  public static final String setToBeginOfLineAction= "setToBeginOfLine";
  /** Name of action to move caret to the prompt. */
  public static final String setToPromptAction= "SetToBegin";
  /** Name of action to move caret to the very end. */
  public static final String setToEndAction= "SetToEnd";
  /** Name of actin to delete the rest of line after caret.*/
  public static final String deleteRestOfLineAction= "KillRestOfLine";
  /** Name of action to move the caret to the begin of next word. */
  public static final String nextWordAction= "NextWord";
  /** Name of action to move the caret to the begin of previous word. */
  public static final String previousWordAction= "PreviousWord";
  /** Name of action when defaultkey typed. */
  public static final String defaultKeyTypedAction= "DefaultKeyTypedAction";
  /** Name of action to move the caret one line upon. */
  public static final String lineUpAction= "LineUp";
  /** Name of action to move the caret one line down. */
  public static final String lineDownAction= "LineDown";
  /** Name of action to insert a new line character. */
  public static final String insertNewLineAction= "InsertNewLine";

  /** contains the default actions _excluding styled actions_. */
  private Action[] DEFAULT_ACTIONS=new DefaultEditorKit().getActions();
  /** contains terminal specific actions including default overrides. */
  static final Action[] TERM_ACTIONS= {
        new EvaluateAction(),
        new CompleteCommandAction(),
        new HistoryPreviousAction(),
        new HistoryNextAction(),
        new KillRestOfLineAction(),
        new SetToEndAction(),
        new SetToLineEndAction(),
        new SetToBeginOfLineAction(),
        new SetToPromptAction(),
        new InsertBreakAction(),
        new CopyAction(),
        new PasteAction(),
        new LeftAction(),
        new NextWordAction(nextWordAction),
        new PreviousWordAction(previousWordAction),
        new DefaultKeyTypedAction(),
        new LineUpAction(),
        new LineDownAction()
    };
  static final JTextComponent.KeyBinding[] defaultBindings={
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
          JTermKit.evaluateAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, Event.SHIFT_MASK),
          JTermKit.insertNewLineAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_UP, 0),
          JTermKit.historyPrevAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, 0),
          JTermKit.historyNextAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_LEFT, 0),
          JTermKit.moveLeftAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0),
          JTermKit.setToBeginOfLineAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_TAB, 0),
          JTermKit.completeCommandAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_A, Event.CTRL_MASK),
          JTermKit.setToBeginOfLineAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_E, Event.CTRL_MASK),
          JTermKit.setToEndOfLineAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_K, Event.CTRL_MASK),
          JTermKit.deleteRestOfLineAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_C, Event.CTRL_MASK),
          DefaultEditorKit.copyAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_V, Event.CTRL_MASK),
          DefaultEditorKit.pasteAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_F, Event.ALT_MASK),
          JTermKit.nextWordAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_B, Event.ALT_MASK),
          JTermKit.previousWordAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_E, Event.ALT_MASK),
          JTermKit.setToEndAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_A, Event.ALT_MASK),
          JTermKit.setToPromptAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_UP, Event.CTRL_MASK),
          JTermKit.lineUpAction),
      new JTextComponent.KeyBinding(
          KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, Event.CTRL_MASK),
          JTermKit.lineDownAction),
  };

  public JTermKit() {
  }

  public Document createDefaultDocument()
  {
    return new Session(new DefaultStringEvaluator());
  }

  public String getContentType()
  {
    return "application/x-jterm";
  }

  public Action[] getActions() {
    return TextAction.augmentList(DEFAULT_ACTIONS, TERM_ACTIONS);
  }
  /** Action attached to the left-arrow key. Moves the caret to the left. */
  public static class LeftAction extends TextAction {
    public LeftAction() {
      super(moveLeftAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        if (target != null) {
          if (target.getCaret().getDot()
            > ((Session)target.getDocument()).promptPosition)
            target.getCaret().setDot(target.getCaret().getDot() - 1);
        }
      } catch (Exception ex) {
        System.out.println(ex.getMessage());
      }
    }
  }
  /** Moves the caret to the prompt.*/
  public static class SetToPromptAction extends TextAction {
    public SetToPromptAction() {
      super(setToPromptAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          target.setCaretPosition(
            ((Session)target.getDocument()).promptPosition);
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  /** Moves the caret to the end of line. */
  public static class SetToLineEndAction extends TextAction {
    public SetToLineEndAction() {
      super(setToEndOfLineAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        Session s= (Session)target.getDocument();
        target.setCaretPosition(
          Utilities.getRowEnd(target, target.getCaret().getDot()));
      } catch (Exception ble) {
        System.out.println(ble.getMessage());
      }
    }
  }
  /** Moves the caret to the end of line. */
  public static class SetToEndAction extends TextAction {
    public SetToEndAction() {
      super(setToEndAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        if (target != null)
          target.setCaretPosition(target.getDocument().getLength());
      } catch (Exception ble) {
        System.out.println(ble.getMessage());
      }
    }
  }
  /** Moves the caret to the begin of line. */
  public static class SetToBeginOfLineAction extends TextAction {
    public SetToBeginOfLineAction() {
      super(setToBeginOfLineAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        Session s= (Session)target.getDocument();
        int tmp= Utilities.getRowStart(target, target.getCaret().getDot());
        if (tmp <= s.promptPosition)
          target.setCaretPosition(s.promptPosition);
        else
          target.setCaretPosition(tmp);
      } catch (Exception ble) {
        System.out.println(ble.getMessage());
      }
    }
  }
  /** Delete all characters after caretPostion. */
  public static class KillRestOfLineAction extends TextAction {
    public KillRestOfLineAction() {
      super(deleteRestOfLineAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        if (target != null) {
          int x= target.getCaret().getDot();
          int tmp= Utilities.getRowEnd(target, x);
          target.getDocument().remove(x, tmp - x);
        }
      } catch (BadLocationException ble) {
        System.out.println("unable to set caret");
      }
    }
  }
  /** Means going backwards in the command history. */
  public static class HistoryPreviousAction extends TextAction {
    public HistoryPreviousAction() {
      super(historyPrevAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          ((Session)target.getDocument()).historyPrevious();
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  /** Means going forward in the command history. */
  public static class HistoryNextAction extends TextAction {
    public HistoryNextAction() {
      super(historyNextAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          ((Session)target.getDocument()).historyNext();
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  /** Triggers evaluation of the current command. */
  public static class EvaluateAction extends TextAction {
    public EvaluateAction() {
      super(evaluateAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      try {
        if (target != null) {
           ((Session)target.getDocument()).evaluate();
           target.setCaretPosition(target.getDocument().getLength());
        }
      } catch (Exception ex) {
        System.out.println(ex.getMessage());
      }
    }
  }
  /** Action attached to the Tab key. Asks for help in completing the
  current command. */
  public static class CompleteCommandAction extends TextAction {
    public CompleteCommandAction() {
      super(completeCommandAction);
    }
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          Session s= (Session)target.getDocument();
          Caret caret= target.getCaret();
          String script= s.getCommand();
          String[] result= s.evaluator.completeCommand(script);
          if (result.length == 0) {
            Toolkit.getDefaultToolkit().beep();
          } else {
            if (result.length == 1) {
              //System.out.println("result[0] is >" + result[0] +"<");
              //System.out.println("WURDE GERUFFEN");
              caret.setDot(s.getLength());
              try {
                s.insertString(caret.getDot(), result[0], s.inputStyle);
              } catch (BadLocationException ble) {
                System.out.println("could not append completion");
              }
            } else {
              // Show ambiguous
              StringBuffer sb= new StringBuffer("\n");
              int i;
              for (i= 0; i < result.length && i < Session.SHOW_AMBIG_MAX; i++)
                sb.append(result[i] + "\n");
              if (i == Session.SHOW_AMBIG_MAX)
                sb.append("...\n");
              s.displayAndPrompt(sb.toString(), s.completionStyle);
              try {
                s.insertString(caret.getDot(), script, s.inputStyle);
              } catch (BadLocationException ble) {
                System.out.println("unable to append completion");
              }
            }
          }
          // force scrolling to the end
          target.setCaretPosition(s.getLength());
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  /** Action that set caret to the next word. */
  public static class NextWordAction extends TextAction {
    /** 
     * Create this action with the appropriate identifier. 
     * @param nm  the name of the action, Action.NAME.
    */
    NextWordAction(String nm) {
      super(nm);
    }
    /** The operation to perform when this action is triggered. */
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        int offs= target.getCaretPosition();
        boolean failed= false;
        try {
          offs= Utilities.getNextWord(target, offs);
        } catch (BadLocationException bl) {
          int end= target.getDocument().getLength();
          if (offs != end) {
            offs= end;
          } else {
            failed= true;
          }
        }
        if (!failed) {
          target.setCaretPosition(offs);
        } else {
          // UIManager.getLookAndFeel().provideErrorFeedback(target);
        }
      }
    }
  }
  /** Action that set caret to the previous word. */
  public static class PreviousWordAction extends TextAction {
    /** 
     * Create this action with the appropriate identifier. 
     * @param nm  the name of the action, Action.NAME.
     */
    PreviousWordAction(String nm) {
      super(nm);
    }
    /** The operation to perform when this action is triggered. */
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        int offs= target.getCaretPosition();
        boolean failed= false;
        try {
          offs= Utilities.getPreviousWord(target, offs);
        } catch (BadLocationException bl) {
          if (offs != 0) {
            offs= 0;
          } else {
            failed= true;
          }
        }
        if (!failed) {
          target.setCaretPosition(offs);
        } else {
          //UIManager.getLookAndFeel().provideErrorFeedback(target);
        }
      }
    }
  }
  /**
   * The action that is executed by default if 
   * a <em>key typed event</em> is received and there
   * is no keymap entry.  There is a variation across
   * different VM's in what gets sent as a <em>key typed</em>
   * event, and this action tries to filter out the undesired
   * events.  This filters the control characters and those
   * with the ALT modifier.  It allows Control-Alt sequences
   * through as these form legitimate unicode characters on
   * some PC keyboards.
   * <p>
   * If the event doesn't get filtered, it will try to insert
   * content into the text editor.  The content is fetched
   * from the command string of the ActionEvent.  The text
   * entry is done through the <code>replaceSelection</code>
   * method on the target text component.  This is the
   * action that will be fired for most text entry tasks.
   * <p>
   * <strong>Warning:</strong>
   * Serialized objects of this class will not be compatible with
   * future Swing releases. The current serialization support is
   * appropriate for short term storage or RMI between applications running
   * the same version of Swing.  As of 1.4, support for long term storage
   * of all JavaBeans<sup><font size="-2">TM</font></sup>
   * has been added to the <code>java.beans</code> package.
   * Please see {@link java.beans.XMLEncoder}.
   *
   * @see DefaultEditorKit#defaultKeyTypedAction
   * @see DefaultEditorKit#getActions
   * @see Keymap#setDefaultAction
   * @see Keymap#getDefaultAction
   */
  public static class DefaultKeyTypedAction extends TextAction {
    /**
     * Creates this object with the appropriate identifier.
     */
    public DefaultKeyTypedAction() {
      super(defaultKeyTypedAction);
    }
    /**
     * The operation to perform when this action is triggered.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if ((target != null) && (e != null)) {
        if ((!target.isEditable()) || (!target.isEnabled())) {
          return;
        }
        if (target.getCaret().getDot()
          < ((Session)target.getDocument()).promptPosition)
          target.getCaret().setDot(target.getDocument().getLength());
        String content= e.getActionCommand();
        int mod= e.getModifiers();
        if ((content != null)
          && (content.length() > 0)
          && ((mod & ActionEvent.ALT_MASK) == (mod & ActionEvent.CTRL_MASK))) {
          char c= content.charAt(0);
          if ((c >= 0x20) && (c != 0x7F)) {
            target.replaceSelection(content);
          }
        }
      }
    }
  }
  /**
   * Action to move the caret to upon line if allowed.
   */
  public static class LineUpAction extends TextAction {
    LineUpAction() {
      super(lineUpAction);
    }
    /** The operation to perform when this action is triggered. */
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          int caretPos= target.getCaret().getDot();
          int lineSet= caretPos - Utilities.getRowStart(target, caretPos);
          int aboveBegin= Utilities.getPositionAbove(target, caretPos, 0);
          int aboveEnd= Utilities.getRowEnd(target, aboveBegin);
          int tmp= aboveBegin + lineSet;
          //    if( tmp >= ((Session)target.getDocument()).promptPosition ){
          if (tmp > aboveEnd)
            target.setCaretPosition(aboveEnd);
          else
            target.setCaretPosition(tmp);
          // }
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  /**
   * Action to move the caret to downon line if allowed.
   */
  public static class LineDownAction extends TextAction {
    LineDownAction() {
      super(lineDownAction);
    }
    /** The operation to perform when this action is triggered. */
    public void actionPerformed(ActionEvent e) {
      JTextComponent target= getTextComponent(e);
      if (target != null) {
        try {
          int caretPos= target.getCaret().getDot();
          int lineSet= caretPos - Utilities.getRowStart(target, caretPos);
          int belowBegin= Utilities.getPositionBelow(target, caretPos, 0);
          int belowEnd= Utilities.getRowEnd(target, belowBegin);
          int tmp= belowBegin + lineSet;
          if (tmp > belowEnd)
            target.setCaretPosition(belowEnd);
          else
            target.setCaretPosition(tmp);
        } catch (Exception ex) {
          System.out.println(ex.getMessage());
        }
      }
    }
  }
  public static class InsertNewLineAction extends TextAction {

      public InsertNewLineAction() {
          super(insertNewLineAction);
      }

      /**
       * Insert a newline &quot;&#x5c;n&quot;.
       */
      public void actionPerformed(ActionEvent e) {
          JTextComponent target = getTextComponent(e);
          if((target != null) && (e != null)) {
              if((! target.isEditable()) || (! target.isEnabled())) {
                  return;
              }
              if (target.getCaret().getDot()
                < ((Session)target.getDocument()).promptPosition)
                target.getCaret().setDot(target.getDocument().getLength());
              String content= e.getActionCommand();
              int mod= e.getModifiers();
              if((mod & ActionEvent.ALT_MASK) == (mod & ActionEvent.CTRL_MASK)) {
                  target.replaceSelection("\n");
              }
          }
      }
  }

}
