/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public abstract class AbstractDialog extends JDialog {
	
	private static final long serialVersionUID = 5543689665195622291L;
	public static int BUTTON_NONE = 0;
	public static int BUTTON_OKAY = 1;
	public static int BUTTON_APPLY = 2;
	public static int BUTTON_CANCEL = 4;
	
	private ChangeListener changeListener;
	private ActionListener okListener;
	private ActionListener cancelListener;
	
	protected ApplyAction acApply = new ApplyAction();
	protected CancelAction acCancel = new CancelAction();
	protected OkayAction acOkay = new OkayAction();
	protected JPanel jpButtons;
	protected JPanel jpMain;
	final private ChangeEvent changeEvent = new ChangeEvent(this);
	
	public AbstractDialog(
			int buttons,
			JComponent component, 
			boolean exitOnEnter
	) {
		jpMain = new JPanel();
		if (component != null) {
			setMainComponent(component);
		}
		jpButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		if ((buttons & BUTTON_OKAY) > 0) {
			JButton jbOkay = new JButton(acOkay);
			jpButtons.add(jbOkay);
		}
		if ((buttons & BUTTON_APPLY) > 0) {
			jpButtons.add(new JButton(acApply));
		}
		if ((buttons & BUTTON_CANCEL) > 0) {
			JButton jbCancel = new JButton(acCancel);
			jpButtons.add(jbCancel);
		}
		
		JPanel jpFrame = new JPanel(new BorderLayout());
		jpFrame.setBorder(new EmptyBorder(5, 5, 5, 5));
		jpFrame.add(jpMain, BorderLayout.CENTER);
		jpFrame.add(jpButtons, BorderLayout.SOUTH);
		
		getContentPane().setLayout(new BorderLayout());
		getContentPane().add(jpFrame, BorderLayout.CENTER);
		
		setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
		setAlwaysOnTop(true);
		addWindowListener(new CloseListener());
		
		if (component != null) {
			pack();
		}
	}
	
	public abstract void setValue(Object o);
	
	public abstract Object getValue();
	
	public AbstractDialog(int buttons, JComponent component) {
		this(buttons, component, true);
	}
	
	public AbstractDialog(int buttons) {
		this(buttons, null);
	}
	
	public AbstractDialog(int buttons, boolean exitOnEnter) {
		this(buttons, null, exitOnEnter);
	}
	
	public AbstractDialog() {
		this(BUTTON_OKAY | BUTTON_CANCEL);
	}

	public void apply() {
	}
	
	public void close() {
		setVisible(false);
	}
	
	public JPanel getButtonPanel() {
		return jpButtons;
	}
	
	public void setMainComponent(Component component) {
		getMainPanel().setLayout(new BorderLayout());
		getMainPanel().add(component, BorderLayout.CENTER);
	}
	
	public JPanel getMainPanel() {
		return jpMain;
	}
	
	public void show(Component c) {
		if (c != null) {
			setLocationRelativeTo(c);
		}
		setVisible(true);
	}
	
	protected void fireStateChanged() {
		if (changeListener != null) {
			changeListener.stateChanged(changeEvent);
		}
	}
	
	public void setChangeListener(ChangeListener changeListener) {
		this.changeListener = changeListener;
	}

	private class ApplyAction extends AbstractAction {
		private static final long serialVersionUID = 3869645622411010713L;

		public ApplyAction() {
			putValue(Action.NAME, "Apply");
			putValue(Action.SHORT_DESCRIPTION, "Applies changes.");
		}
		
		public void actionPerformed(ActionEvent event) {
			try {
				apply();
			}
			catch (IllegalArgumentException e) {
				JOptionPane.showMessageDialog
				(AbstractDialog.this, e.getMessage(), 
						"Illegal Value", JOptionPane.ERROR_MESSAGE);
			}
		}
	}

	private class CancelAction extends AbstractAction {
		private static final long serialVersionUID = -1881501043731518772L;

		public CancelAction() {
			putValue(Action.NAME, "Cancel");
			putValue(Action.SHORT_DESCRIPTION, "Closes the dialog without saving changes.");
		}
		
		public void actionPerformed(ActionEvent e) {
			if (cancelListener != null) cancelListener.actionPerformed(e);
			setVisible(false);
		}
	}
	
	private class OkayAction extends AbstractAction {
		private static final long serialVersionUID = -3866052681450602086L;

		public OkayAction() {
			putValue(Action.NAME, "OK");
			putValue(Action.SHORT_DESCRIPTION, "Closes the dialog saving changes.");
			putValue(Action.MNEMONIC_KEY, new Integer('O'));
		}
		
		public void actionPerformed(ActionEvent event) {
			final ActionEvent e = new ActionEvent(this,0,"ok");
			if (okListener != null) okListener.actionPerformed(e);
			setVisible(false);
		}
	}
	
	private class CloseListener extends WindowAdapter {
		public void windowClosing (java.awt.event.WindowEvent evt) {
			final ActionEvent e = new ActionEvent(this,0,"cancel");
			if (cancelListener != null) cancelListener.actionPerformed(e);
		}
	}

	public void setCancelListener(ActionListener cancelListener) {
		this.cancelListener = cancelListener;
	}

	public void setOkListener(ActionListener okListener) {
		this.okListener = okListener;
	}
}