/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2002-2009, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.jterm;

import java.lang.reflect.Method;

import bsh.BshClassManager;
import bsh.EvalError;
import bsh.Interpreter;
import bsh.NameSpace;
import bsh.TargetError;
import bsh.classpath.BshClassPath;
import bsh.util.NameCompletionTable;

/** Wrapper around a Beanshell Java interpreter. */
public class BshEvaluator implements StringEvaluator {

    private Interpreter interpreter;
    private NameCompletionTable nameCompletion;
    private NameSpace nameSpace;

    /** creates an Evaluator with its own interpreter. */
    public BshEvaluator() {
        this(new Interpreter());
    }

    /** creates an Evaluator that talks to the global namespace of the
        Beanshell interpreter i. */
    public BshEvaluator(Interpreter i) {
        this(i, i.getNameSpace());
    }
 
    /** creates an Evaluator that talks to the namespace ns of the
        Beanshell interpreter i. */
    public BshEvaluator(Interpreter i, NameSpace ns) {
        interpreter = i;
        nameSpace = ns;
        // To prevent crash if name Completion is attempted before the first
        // evaluation
        try {
          //interpreter.eval("print(\"ready\")");
          interpreter.eval("42+0.815");
        } catch(EvalError e) {
           System.out.println(e.getErrorText());
        }
        nameCompletion = new NameCompletionTable();
        nameCompletion.add(nameSpace);
        BshClassManager bcm = interpreter.getClassManager();
        // Hack because only the subclass ClassManagerImpl of
        // BshClassManager knows about getClassPath (therefore
        // we use an auxiliary intrepreter).
        // NOTE: the recommended? way of using an aux interpreter is
        // now replaced by reflection code, it's still some sort of...
        if ( bcm != null ) try {
          Method m=bcm.getClass().getDeclaredMethod("getClassPath", null);
          m.setAccessible(true);
          nameCompletion.add((BshClassPath)m.invoke(bcm, null));
        } catch(Exception ex) {
          ex.printStackTrace();
        }
    }

    public Interpreter getInterpreter()
    {
        return interpreter;
    }

    public NameCompletionTable getNameCompletion()
    {
        return nameCompletion;
    }

    public InterpreterResult evaluate(String script) {
        try {
            Object obj = interpreter.eval(script, nameSpace);
            if (obj == null) {
                return(new InterpreterResult("null"));
            } else {
                return(new InterpreterResult(obj.toString()));
            }
        } catch(TargetError ee) {
            return new InterpreterResult(ee.getTarget().toString(), true);
        } catch(EvalError ee) {
            return new InterpreterResult(ee.getMessage(), true);
        }
    }

    public String [] completeCommand( String part ) {
        int i=part.length()-1;

        // Code portion from bsh.util.JConsole
        while ( 
               i >= 0 && 
               ( Character.isJavaIdentifierPart(part.charAt(i)) 
                 || part.charAt(i) == '.' )
               ) 
            i--;

        part = part.substring(i+1);

        if ( part.length() < 2 )  { // reasonable completion length
            String[] result = new String[0];
            java.awt.Toolkit.getDefaultToolkit().beep();
            return(result);
        }
        String [] complete = nameCompletion.completeName(part);
        // Found one completion, extract the piece to append
        if ( complete.length == 1 ) {
            complete[0] = complete[0].substring(part.length());
        }
        return(complete);
    }
}
