/**
This file is part of a jTEM project.
All jTEM projects are licensed under the FreeBSD license 
or 2-clause BSD license (see http://www.opensource.org/licenses/bsd-license.php). 

Copyright (c) 2006-2010, Technische Universität Berlin, jTEM
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, 
are permitted provided that the following conditions are met:

-	Redistributions of source code must retain the above copyright notice, 
	this list of conditions and the following disclaimer.

-	Redistributions in binary form must reproduce the above copyright notice, 
	this list of conditions and the following disclaimer in the documentation 
	and/or other materials provided with the distribution.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY 
OF SUCH DAMAGE.
**/

package de.jtem.beans;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.URL;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class DimensionPanel extends JPanel {

	private static final long serialVersionUID = -689497176051389498L;
	
	private SpinnerNumberModel widthModel;
	private SpinnerNumberModel heightModel;
	private JButton keepAspectButton;
	private boolean keepAspect = true;
	private List<ChangeListener> changeListeners = new CopyOnWriteArrayList<ChangeListener>();
	private double aspectRatio;
	private boolean doNotFire;
	private ImageIcon openChain;
	private ImageIcon closedChain;
	
	public DimensionPanel() {
		super (new GridBagLayout());
		widthModel = new SpinnerNumberModel(
				new Integer(1),
				new Integer(1),
				null,
				new Integer(1)
		);
		widthModel.addChangeListener(new ChangeListener() {

			public void stateChanged(ChangeEvent e) {
				setWidthFromSpinner();
			}
		});
		heightModel = new SpinnerNumberModel(
				new Integer(1),
				new Integer(1),
				null,
				new Integer(1)
		);
		heightModel.addChangeListener(new ChangeListener() {

			public void stateChanged(ChangeEvent e) {
				setHeightFromSpinner();
			}
		});
		
		Insets insets = new Insets(3,3,3,3);
		JLabel widthLabel = new JLabel("width:");
		widthLabel.setHorizontalAlignment(JLabel.RIGHT);
		GridBagConstraints gbc = new GridBagConstraints();
		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.insets = insets;
		gbc.anchor = GridBagConstraints.EAST;
		gbc.fill = GridBagConstraints.BOTH;
		add(widthLabel, gbc);
		
		JSpinner widthSpinner = new JSpinner(widthModel);
		widthSpinner.setPreferredSize(new Dimension(60,25));
		gbc = new GridBagConstraints();
		gbc.gridx = 1;
		gbc.gridy = 0;
		gbc.insets = insets;
		gbc.fill = GridBagConstraints.BOTH;
		add(widthSpinner, gbc);
		
		JLabel heightLabel = new JLabel("height:");
		heightLabel.setHorizontalAlignment(JLabel.RIGHT);
		gbc = new GridBagConstraints();
		gbc.gridx = 0;
		gbc.gridy = 1;
		gbc.insets = insets;
		gbc.anchor = GridBagConstraints.EAST;
		gbc.fill = GridBagConstraints.BOTH;
		add(heightLabel, gbc);
		
		JSpinner heightSpinner = new JSpinner(heightModel);
		heightSpinner.setPreferredSize(new Dimension(60,25));
		gbc = new GridBagConstraints();
		gbc.gridx = 1;
		gbc.gridy = 1;
		gbc.insets = insets;
		gbc.fill = GridBagConstraints.BOTH;
		add(heightSpinner, gbc);
		
		keepAspectButton = new JButton();
		//keepAspectButton.setBorder(null);
		keepAspectButton.setFocusable(false);
		keepAspectButton.setMargin(insets);
		URL imgURL = DimensionPanel.class.getResource("openChain.gif");
		openChain = new ImageIcon(imgURL);
		imgURL = DimensionPanel.class.getResource("closedChain.gif");
		closedChain = new ImageIcon(imgURL);
		keepAspectButton.setIcon(keepAspect ? closedChain : openChain);
		keepAspectButton.addActionListener(new ActionListener() {

			public void actionPerformed(ActionEvent e) {
				keepAspect = !keepAspect;
				keepAspectButton.setIcon(keepAspect ? closedChain : openChain);
			}
		});
		gbc = new GridBagConstraints();
		gbc.gridy = 0;
		gbc.gridx = 2;
		gbc.gridheight = 2;
		gbc.insets = insets;
		gbc.fill = GridBagConstraints.BOTH;
		add(keepAspectButton, gbc);
	}
	
	public Dimension getDimension() {
		return new Dimension(
				widthModel.getNumber().intValue(),
				heightModel.getNumber().intValue()
		);
	}
	
	public void setDimension(Dimension d) {
		int w = widthModel.getNumber().intValue();
		int h = heightModel.getNumber().intValue();
		aspectRatio = d.height/(double)d.width;
		if (d.width != w || d.height != h) {
			doNotFire = true;
			widthModel.setValue(new Integer(d.width));
			heightModel.setValue(new Integer(d.height));
			doNotFire = false;
			fireChange();
		}
	}
	
	private void setWidthFromSpinner() {
		int w = widthModel.getNumber().intValue();
		if (!doNotFire) {
			if (keepAspect) {
				doNotFire = true;
				heightModel.setValue(new Integer((int) (w*aspectRatio)));
				doNotFire = false;
			} else {
				double h = heightModel.getNumber().doubleValue();
				aspectRatio = h/w;
			}
		}
		fireChange();
	}
	
	private void setHeightFromSpinner() {
		int h = heightModel.getNumber().intValue();
		if (!doNotFire) {
			if (keepAspect) {
				doNotFire = true;
				widthModel.setValue(new Integer((int) (h/aspectRatio)));
				doNotFire = false;
			} else {
				double w = widthModel.getNumber().doubleValue();
				aspectRatio = h/w;
			}
		}
		fireChange();
	}
	
	private void fireChange() {
		for (ChangeListener changeListener : changeListeners) {
			changeListener.stateChanged(new ChangeEvent(this));
		}
	}
	
	public void addChangeListener(ChangeListener listener) {
		changeListeners.add(listener);
	}

	public void removeChangeListener(ChangeListener listener) {
		changeListeners.remove(listener);
	}
}
