// -*- C++ -*-

/* This software was produced by NIST, an agency of the U.S. government,
 * and by statute is not subject to copyright in the United States.
 * Recipients of this software assume all responsibilities associated
 * with its operation, modification and maintenance. However, to
 * facilitate maintenance we ask that before distributing modified
 * versions of this software, you first contact the authors at
 * oof_manager@nist.gov. 
 */

#include <oofconfig.h>
#include <iostream>

#ifndef PYTHONLOCK_H
#define PYTHONLOCK_H

// Classes for handling the python global interpreter lock and thread
// state.

// This scheme was copied from code generated by swig 4.0.2 (by
// running swig with the -threads option and examining the C++
// output).  This version is modified slightly so that the
// Python_Thread_Allow and Python_Thread_Block objects do their work
// in their start() methods, not in their constructors.  This allows
// threading to be turned on and off at run time, instead of compile
// time.  The original swig code is meant to be used like this:
//
//    SWIG_Python_Thread_Allow allow_thread;
//    function_call()
//    allow_thread.end()
//
// but we might need to use it like this:
//
//    Python_Thread_Allow allow_thread(false); // false means "don't start".
//    if(something)
//       allow_thread.start();
//    function_call();
//    if(something)
//      allow_thread.end()
//
// Python_Thread_Allow.start() allows *other* threads to run.  It
// releases the global interpreter lock, and should be called when a
// python thread is going to do something in C++ that takes a long
// time.  The swig %exception typemap uses it to wrap all calls to
// C++.
//
// Python_Thread_Block.start() acquires the lock, and should be called
// when C++ needs to make calls to the Python API.
//
// The constructors for Python_Thread_Allow and Python_Thread_Block
// take an optional boolean argument. If it's true, then their
// constructors call the start() method.  The default is true.

extern bool threading_enabled;

class Python_Thread_Block {
private:
  bool status;
  PyGILState_STATE state;
public:
  void start() {
    if(!status && threading_enabled) {
      //std::cerr << "OOF::Python_Thread_Block: start"  << std::endl;
      state = PyGILState_Ensure();
      status = true;
    }
  }
  void end() {
    if (status && threading_enabled) {
      //std::cerr << "OOF::Python_Thread_Block: end" << std::endl;
      PyGILState_Release(state);
      status = false;
    }
  }
  Python_Thread_Block(bool on=true)
    : status(false)
  {
    if(on)
      start();
  }
  ~Python_Thread_Block() {
    end();
  }
};

class Python_Thread_Allow {
private:
  PyThreadState *save;
public:
  Python_Thread_Allow(bool on=true)
    : save(nullptr)
  {
    if(on)
      start();
  }
  ~Python_Thread_Allow() {
    end();
  }
  void end() {
    if(save) {
      //std::cerr << "OOF::Python_Thread_Allow: end" << std::endl;
      PyEval_RestoreThread(save);
      save = nullptr;
    }
  }
  void start() {
    if(!save && threading_enabled) {
      //std::cerr << "OOF::Python_Thread_Allow: start" << std::endl;
      save = PyEval_SaveThread();
    }
  }
};

// These macros are useful only if the begin and end calls are in the
// same scope.
#define PYTHON_THREAD_BEGIN_BLOCK   Python_Thread_Block _thread_block(true)
#define PYTHON_THREAD_END_BLOCK     _thread_block.end()
#define PYTHON_THREAD_BEGIN_ALLOW   Python_Thread_Allow _thread_allow(true)
#define PYTHON_THREAD_END_ALLOW     _thread_allow.end()


#endif // PYTHONLOCK_H
