#!/usr/bin/env python
#
# Create project files for all BTstack embedded examples in WICED/apps/btstack

import os
import shutil
import sys
import time
import subprocess

mk_template = '''#
# BTstack example 'EXAMPLE' for WICED port
#
# Generated by TOOL
# On DATE

NAME := EXAMPLE

GLOBAL_INCLUDES += .

# Replace Linefeed with -> CRLF
GLOBAL_DEFINES += CRLF_STDIO_REPLACEMENT

$(NAME)_SOURCES := ../../../libraries/btstack/example/EXAMPLE.c
$(NAME)_COMPONENTS += btstack/port/wiced-h4

# micro-ecc of WICED tree used for SECP256R1 in LE Secure Connections
MICRO_ECC := MICRO_ECC_COMPONENT

# Additional CFLAGS for BTstack Component compilation
BTSTACK_CFLAGS += ADDITIONAL_CFLAGS

# Name of Firmware file
BT_FIRMWARE_FILE := BLUETOOTH_FIRMWARE_FILE
'''

gatt_update_template = '''#!/bin/sh
DIR=`dirname $0`
BTSTACK_ROOT=$DIR/../../../libraries/btstack
echo "Creating EXAMPLE.h from EXAMPLE.gatt"
$BTSTACK_ROOT/tool/compile_gatt.py $BTSTACK_ROOT/example/EXAMPLE.gatt $DIR/EXAMPLE.h
'''

# get script path
script_path = os.path.abspath(os.path.dirname(sys.argv[0]))

# validate WICED root by reading version.txt
wiced_root = script_path + "/../../../../"
wiced_version_txt = ""
try:
    with open(wiced_root + 'version.txt', 'r') as fin:
        wiced_version_txt = fin.read()  # Read the contents of the file into memory.
except:
    pass
if not "WICED Version" in wiced_version_txt:
    print("Cannot find WICED root. Make sure BTstack is checked out in WICED-SDK-X/libraries")
    sys.exit(1)

# check for 5.2+ version syntax
if 'Wiced_' in wiced_version_txt:
    wiced_version_string = (wiced_version_txt.split()[2]).split('_')[1]
    wiced_version_major = int(wiced_version_string.split('.')[0])
    wiced_version_minor = int(wiced_version_string.split('.')[1])
else:
    wiced_version = wiced_version_txt.split()[2]
    wiced_version_major = int(wiced_version.split('.')[0])
    wiced_version_minor = int(wiced_version.split('.')[1])

wiced_version = "%u.%u" % (wiced_version_major, wiced_version_minor)

# show WICED version
print("\nFound WICED SDK version: %s" % wiced_version)

additional_cflags = ""
if wiced_version < "3.4.0":
    print("Adding WICED_UART_READ_DOES_NOT_RETURN_BYTES_READ for SDK < 3.4.0")
    additional_cflags = "-DWICED_UART_READ_DOES_NOT_RETURN_BYTES_READ"

# NOTE: it would be more robust to check for files on disk

# bluetooth firmware image name changed in 5.2
if wiced_version < "5.2":
    bluetooth_firmware_file = 'bt_firmware_image.c'
else:
    bluetooth_firmware_file = 'bt_firmware_controller.c'
print("Bluetooth Firmware name: %s" % bluetooth_firmware_file)

# micro-ecc moved in 6.2 from libraries/crypto/micro-ecc to WICED/security/BESL/crypto_internal/micro-ecc
if wiced_version < "6.2":
    micro_ecc_component = "crypto/micro-ecc"
else:
    micro_ecc_component = "BESL/crypto_internal/micro-ecc"
print("micro-ecc component: %s" % micro_ecc_component)

# path to examples
examples_embedded = script_path + "/../../example/"

# path to WICED/apps/btstack
apps_btstack = wiced_root + "/apps/btstack/"

print("\nCreating examples in apps/btstack:")

# iterate over btstack examples
for file in os.listdir(examples_embedded):
    if not file.endswith(".c"):
        continue
    if file in ['panu_demo.c', 'sco_demo_util.c', 'ant_test.c', 'pan_lwip_http_server.c']:
        continue
    example = file[:-2]

    # create folder
    apps_folder = apps_btstack + example + "/"
    if not os.path.exists(apps_folder):
        os.makedirs(apps_folder)

    # create .mk file
    with open(apps_folder + example + ".mk", "wt") as fout:
        fout.write(mk_template.replace("EXAMPLE", example)
            .replace("TOOL", script_path)
            .replace("ADDITIONAL_CFLAGS", additional_cflags)
            .replace("DATE",time.strftime("%c"))
            .replace('BLUETOOTH_FIRMWARE_FILE', bluetooth_firmware_file)
            .replace('MICRO_ECC_COMPONENT', micro_ecc_component))

    # create update_gatt.sh if .gatt file is present
    gatt_path = examples_embedded + example + ".gatt"
    if os.path.exists(gatt_path):
        update_gatt_script = apps_folder + "update_gatt_db.sh"
        with open(update_gatt_script, "wt") as fout:
            fout.write(gatt_update_template.replace("EXAMPLE", example))        
        os.chmod(update_gatt_script, 0o755)
        subprocess.call(update_gatt_script + "> /dev/null", shell=True)
        print("- %s including compiled GATT DB" % example)
    else:
        print("- %s" % example)

