/* filehash:  print various hash digests and filesizes for specified files

   Copyright (C) 2004-2025 by Brian Lindholm.  This file is part of the
   littleutils utility set.

   The filehash utility is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3, or (at your option) any later
   version.

   The filehash utility is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   the littleutils.  If not, see <https://www.gnu.org/licenses/>. */


#include <config.h>

#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#include <limits.h>
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
# define OPTEND -1
#else
# define OPTEND EOF
#endif
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

#ifdef MSDOS
# include "io.h"
#endif

#include "md5.h"
#include "sha1.h"
#include "sha256.h"
#include "sha512.h"
#include "b2sum.h"

#ifdef HAVE_FSEEKO
# define fseek(a,b,c) fseeko((a),(off_t)(b),(c))
#endif

#ifdef __MINGW32__
extern int getopt (int argc, char * const *argv, const char *optstring);
extern char *optarg;
extern int optind;
#endif

#ifdef DJGPP
unsigned short _djstat_flags = 63;
#endif

#ifndef PATH_MAX
# define PATH_MAX 256
#endif

char *sizefmt = (sizeof (off_t) <= sizeof (long) ? "%lu" : "%llu");


static void
help (FILE *where)
{
  fprintf (where,
    "filehash " PACKAGE_VERSION "\n"
    "usage: filehash [-1(MD5)] [-2(SHA1)] [-3(SHA224) ][-4(SHA256)]\n"
    "         [-5(SHA384)] [-6(SHA512)] [-7(BLAKE2B_256)] [-8(BLAKE2B_512)]\n"
    "         [-B(ase64)] [-b(ase64url)] [-C(lassic_BSD)] [-c(lassic)]\n"
    "         [-f file_list] [-h(elp)] [-n byte_count] [-o offset] [-p(ipe)]\n"
    "         [-q(uiet)] [-s(ize)] [-v(erbose)] file...\n");
}


static void
encode_hash (unsigned char *string, unsigned char *hash, int bytes, int use_base64)
{
  int i, j;
  static const unsigned char base64url_table[65] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
  static const unsigned char base64_table[65] =
    "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
  static const unsigned char hex_table[17] = "0123456789abcdef";

  j = 0;
  if (use_base64 == 2) {
    for (i = 0; i < (bytes / 3); ++i) {
      string[j++] = base64url_table[hash[i*3] >> 2];
      string[j++] = base64url_table[((hash[i*3] & 0x03) << 4) | (hash[i*3+1] >> 4)];
      string[j++] = base64url_table[((hash[i*3+1] & 0x0F) << 2) | (hash[i*3+2] >> 6)];
      string[j++] = base64url_table[hash[i*3+2] & 0x3F];
    }
    if ((bytes % 3) == 2) {
      string[j++] = base64url_table[hash[bytes-2] >> 2];
      string[j++] = base64url_table[((hash[bytes-2] & 0x03) << 4) | (hash[bytes-1] >> 4)];
      string[j++] = base64url_table[(hash[bytes-1] & 0x0F) << 2];
    }
    else if ((bytes % 3) == 1) {
      string[j++] = base64url_table[hash[bytes-1] >> 2];
      string[j++] = base64url_table[(hash[bytes-1] & 0x03) << 4];
    }
    string[j++] = '\000';
  }
  else if (use_base64 == 1) {
    for (i = 0; i < (bytes / 3); ++i) {
      string[j++] = base64_table[hash[i*3] >> 2];
      string[j++] = base64_table[((hash[i*3] & 0x03) << 4) | (hash[i*3+1] >> 4)];
      string[j++] = base64_table[((hash[i*3+1] & 0x0F) << 2) | (hash[i*3+2] >> 6)];
      string[j++] = base64_table[hash[i*3+2] & 0x3F];
    }
    if ((bytes % 3) == 2) {
      string[j++] = base64_table[hash[bytes-2] >> 2];
      string[j++] = base64_table[((hash[bytes-2] & 0x03) << 4) | (hash[bytes-1] >> 4)];
      string[j++] = base64_table[(hash[bytes-1] & 0x0F) << 2];
    }
    else if ((bytes % 3) == 1) {
      string[j++] = base64_table[hash[bytes-1] >> 2];
      string[j++] = base64_table[(hash[bytes-1] & 0x03) << 4];
    }
    string[j++] = '\000';
  }
  else {
    for (i = 0; i < bytes; ++i) {
      string[j++] = hex_table[hash[i] >> 4];
      string[j++] = hex_table[hash[i] & 0x0F];
    }
    string[j++] = '\000';
  }
}


static void
print_filehash (char *filename, off_t offset, off_t read_bytes, int print_size,
  int run_md5, int run_sha1, int run_sha224, int run_sha256, int run_sha384,
  int run_sha512, int run_blake2b_256, int run_blake2b_512, int classic,
  int use_base64, int verbose)
{
  FILE *file;
  int i, not_first, rc;
  struct stat file_stats;
  unsigned char md5result[16], sha1result[20], sha224result[28],
                sha256result[32], sha384result[48], sha512result[64],
                blake2b_256result[32], blake2b_512result[64], string[129];

  if (stat (filename, &file_stats))
    fprintf (stderr, "filehash error: can't stat %s\n", filename);
  else if (((file_stats.st_mode & S_IFDIR) != S_IFDIR) &&
           ((file_stats.st_mode & S_IFREG) == S_IFREG))
    {
      if ((file = fopen (filename, "rb")) == NULL)
        fprintf (stderr, "filehash error: can't open %s\n", filename);
      else
        {
          not_first = 0;
          if (run_md5) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = md5_stream (file, md5result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: md5_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha1) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = sha1_stream (file, sha1result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha1_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha224) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = sha224_stream (file, sha224result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha224_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha256) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = sha256_stream (file, sha256result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha256_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha384) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = sha384_stream (file, sha384result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha384_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_sha512) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = sha512_stream (file, sha512result, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha512_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_blake2b_256) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = blake2b_stream (file, blake2b_256result, 32, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha384_stream failed on %s\n", filename);
            not_first = 1;
          }
          if (run_blake2b_512) {
            if (not_first || (offset > 0))
              (void) fseek (file, (off_t) offset, 0);
            rc = blake2b_stream (file, blake2b_512result, 64, read_bytes);
            if (rc)
              fprintf (stderr, "filehash error: sha512_stream failed on %s\n", filename);
            not_first = 1;
          }
          (void) fclose (file);
          not_first = 0;
          if ((verbose == 1) && (classic == 0)) {
            fprintf (stdout, "%s", filename);
            not_first = 1;
          }
          if (print_size) {
            if (classic == 2) {
              fprintf (stdout, sizefmt, file_stats.st_size);
              fprintf (stdout, "  %s\n", filename);
            }
            else if (classic == 1) {
              fprintf (stdout, "SIZE (%s) = ", filename);
              fprintf (stdout, sizefmt, file_stats.st_size);
              fprintf (stdout, "\n");
            }
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, sizefmt, file_stats.st_size);
            }
          }
          if (run_md5) {
            encode_hash(string, md5result, 16, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "MD5 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_sha1) {
            encode_hash(string, sha1result, 20, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "SHA1 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_sha224) {
            encode_hash(string, sha224result, 28, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "SHA224 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_sha256) {
            encode_hash(string, sha256result, 32, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "SHA256 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_sha384) {
            encode_hash(string, sha384result, 48, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "SHA384 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_sha512) {
            encode_hash(string, sha512result, 64, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "SHA512 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_blake2b_256) {
            encode_hash(string, blake2b_256result, 32, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "BLAKE2b-256 (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (run_blake2b_512) {
            encode_hash(string, blake2b_512result, 64, use_base64);
            if (classic == 2)
              fprintf (stdout, "%s  %s\n", string, filename);
            else if (classic == 1)
              fprintf (stdout, "BLAKE2b (%s) = %s\n", filename, string);
            else {
              if (not_first)
                fprintf (stdout, "\t");
              not_first = 1;
              fprintf (stdout, "%s", string);
            }
          }
          if (classic == 0)
            fprintf (stdout, "\n");
        }
    }
}


int
main (int argc, char **argv)
{
  FILE *infile;
  char filename[PATH_MAX], *listname, *newline, *rc;
  int argn, classic, offset, opt, print_size, run_md5, run_sha1, run_sha224,
      run_sha256, run_sha384, run_sha512, run_blake2b_256, run_blake2b_512,
      use_base64, use_file, use_pipe, verbose;
  off_t read_bytes, tmp_bytes;

  /* parse options */

  classic = 0;
  listname = "";
  offset = 0;
  print_size = 0;
  read_bytes = -1;
  run_md5 = 0;
  run_sha1 = 0;
  run_sha224 = 0;
  run_sha256 = 0;
  run_sha384 = 0;
  run_sha512 = 0;
  run_blake2b_256 = 0;
  run_blake2b_512 = 0;
  use_base64 = 0;
  use_file = 0;
  use_pipe = 0;
  verbose = 0;
  while ((opt = getopt (argc, argv, "12345678BbCcf:hpn:o:qsv")) != OPTEND)
    switch (opt)
      {
      case '1':
        run_md5 = 1;
        break;
      case '2':
        run_sha1 = 1;
        break;
      case '3':
        run_sha224 = 1;
        break;
      case '4':
        run_sha256 = 1;
        break;
      case '5':
        run_sha384 = 1;
        break;
      case '6':
        run_sha512 = 1;
        break;
      case '7':
        run_blake2b_256 = 1;
        break;
      case '8':
        run_blake2b_512 = 1;
        break;
      case 'B':
        use_base64 = 1;
        break;
      case 'b':
        use_base64 = 2;
        break;
      case 'C':
        classic = 1;
        break;
      case 'c':
        classic = 2;
        break;
      case 'f':
        use_file = 1;
        listname = optarg;
        break;
      case 'h':
        help (stdout);
        return (0);
      case 'n':
        tmp_bytes = (sizeof (off_t) <= sizeof (long) ? atol(optarg) : atoll(optarg));
        if (tmp_bytes > 0)
          read_bytes = tmp_bytes;
        else
          read_bytes = 0;
        break;
      case 'o':
        tmp_bytes = (sizeof (off_t) <= sizeof (long) ? atol(optarg) : atoll(optarg));
        if (tmp_bytes > 0)
          offset = tmp_bytes;
        else
          offset = 0;
        break;
      case 'p':
        use_pipe = 1;
        break;
      case 'q':
        verbose = -1;
        break;
      case 's':
        print_size = 1;
        break;
      case 'v':
        verbose = 1;
        break;
      case '?':
        help (stderr);
        return (1);
      }

  /* finalize options */

  if ((optind == argc) && (use_file == 0) && (use_pipe == 0))
    {
      help (stdout);
      return (0);
    }
  if (verbose == 0)
    {
      if (((argc - optind) != 1) || use_file || use_pipe)
        verbose = 1;
      else
        verbose = -1;
    }
  if ((run_md5 + run_sha1 + run_sha224 + run_sha256 + run_sha384 + run_sha512 +
    run_blake2b_256 + run_blake2b_512) == 0)
      run_sha256 = 1;

  /* process files in listed in file specified by -f option */

  if (use_file)
    {
      infile = fopen (listname, "r");
      if (infile == NULL)
        fprintf (stderr, "filehash error: can't open %s!\n", listname);
      else
        {
          while (!feof (infile))
            {
              rc = fgets (filename, PATH_MAX - 1, infile);
              if (rc != NULL)
                {
                  newline = strchr (filename, '\n');
                  if (newline != NULL)
                    *newline = '\0';
                  if (strlen (filename) != 0)
                    print_filehash (filename, offset, read_bytes, print_size,
                      run_md5, run_sha1, run_sha224, run_sha256, run_sha384,
                      run_sha512, run_blake2b_256, run_blake2b_512, classic,
                      use_base64, verbose);
                }
            }
          (void) fclose (infile);
        }
    }

  /* process files listed on stdin (i.e., the -p option) */

  if (use_pipe)
    while (!feof (stdin))
      {
        rc = fgets (filename, PATH_MAX - 1, stdin);
        if (rc != NULL)
          {
            newline = strchr (filename, '\n');
            if (newline != NULL)
              *newline = '\0';
            if (strlen (filename) != 0)
              print_filehash (filename, offset, read_bytes, print_size, run_md5,
                run_sha1, run_sha224, run_sha256, run_sha384, run_sha512,
                run_blake2b_256, run_blake2b_512, classic, use_base64, verbose);
          }
      }

  /* process files given in the argument list */

  for (argn = optind; argn < argc; argn++)
    print_filehash (argv[argn], offset, read_bytes, print_size, run_md5,
      run_sha1, run_sha224, run_sha256, run_sha384, run_sha512,
      run_blake2b_256, run_blake2b_512, classic, use_base64, verbose);

  /* indicate successful finish */

  return (0);
}
