#ifndef FrameCPP_VERSION_8_FrTOCSimEvent_HH
#define FrameCPP_VERSION_8_FrTOCSimEvent_HH

#include "framecpp/Version7/FrTOCSimEvent.hh"

#include "framecpp/Version8/STRING.hh"
#include "framecpp/Version8/GPSTime.hh"

namespace FrameCPP
{
  namespace Common
  {
    class TOCInfo;
  }

  namespace Version_8
  {
    //===================================================================
    /// \brief Indexed elements of the FrSimEvent structure.
    //===================================================================
    class FrTOCSimEvent
    {
    public:
      typedef INT_4U	nevent_type;
      typedef STRING	name_type;
      typedef INT_4U	nTotalSEvent_type;
      typedef INT_4U	gtimesSim_type;
      typedef INT_4U	gtimenSim_type;
      typedef REAL_4	amplitudeSimEvent_type;
      typedef INT_8U	positionSimEvent_type;

      struct event_type {
	//---------------------------------------------------------------
	/// \brief Start time of event.
	//---------------------------------------------------------------
	GPSTime			GTime;
	//---------------------------------------------------------------
	/// \brief Amplitude of event.
	//---------------------------------------------------------------
	amplitudeSimEvent_type	amplitudeSimEvent;
	//---------------------------------------------------------------
	/// \brief FrSimEvent offset positions.
	///
	/// FrSimEvent offset positions, in bytes,
	/// from the beginning of the file.
	//---------------------------------------------------------------
	positionSimEvent_type	positionSimEvent;

	const event_type& operator=( const Previous::FrTOCSimEvent::event_type& Source );
      };

      static const int event_type_size
      = sizeof( gtimesSim_type )
	+ sizeof( gtimenSim_type )
	+ sizeof( amplitudeSimEvent_type )
	+ sizeof( positionSimEvent_type )
	;

      typedef std::vector< event_type > events_container_type;
      typedef std::map< name_type, events_container_type >
      nameSimEvent_container_type;

      //-----------------------------------------------------------------
      /// \brief Default constructor
      //-----------------------------------------------------------------
      FrTOCSimEvent( );

      //-----------------------------------------------------------------
      /// \brief Number of bytes needed to write this structure
      ///
      /// \param[in] Stream
      ///     The stream from which to the object is being read or
      ///     written.
      ///
      /// \return
      ///     The number of bytes need to read or write this object.
      //-----------------------------------------------------------------
      Common::FrameSpec::size_type
      Bytes( const Common::StreamBase& Stream ) const;

      //-----------------------------------------------------------------
      /// \brief Return all events
      ///
      /// \return
      ///     Constant container to the indexed FrSimEvent elements.
      //-----------------------------------------------------------------
      const nameSimEvent_container_type& GetSimEvent( ) const;

      //-----------------------------------------------------------------
      /// \brief Gather TOC info for FrSimEvent being written.
      ///
      /// \param[in] Info
      ///     Information
      ///
      /// \param[in] FrameOffset
      ///     The frame offset of the frame being written.
      ///     The frame offsets start at 0 (zero).
      ///
      /// \param[in] Position
      ///     The byte offset from the start of the file
      ///     where the structure is written.
      //-----------------------------------------------------------------
      void QuerySimEvent( const Common::TOCInfo& Info,
			  INT_4U FrameOffset,
			  INT_8U Position );

      //-----------------------------------------------------------------
      /// \brief asignment operator
      ///
      /// \param[in] Source
      ///     The source to be copied.
      //-----------------------------------------------------------------
      const FrTOCSimEvent& operator=( const Previous::FrTOCSimEvent& Source );

      //-----------------------------------------------------------------
      /// \brief equality operator
      ///
      /// \param[in] RHS
      ///     The FrTOCAdcData object to be compared.
      ///
      /// \return
      ///     The value true is returned if this object is equivelent
      ///     to the RHS instance; false otherwise.
      //-----------------------------------------------------------------
      bool operator==( const FrTOCSimEvent& RHS ) const;

      //-----------------------------------------------------------------
      /// \brief The description of structure
      ///
      /// \param[out] Desc
      ///     Storage for the description of the structure.
      ///
      /// \return
      ///     A Descriptio nobject which describes this structure as
      ///     specified by the frame specification.
      //-----------------------------------------------------------------
      template< typename SE >
      static void Description( Common::Description& Desc );

    protected:
      nameSimEvent_container_type		m_info;

      //-----------------------------------------------------------------
      /// \brief Stream constructor
      ///
      /// \param[in] Stream
      ///     The input stream from where the object is being read.
      //-----------------------------------------------------------------
      FrTOCSimEvent( Common::IStream& Stream );

      //-----------------------------------------------------------------
      /// \brief Locate the position of the event
      ///
      /// \param[in] Frame
      ///     Frame offset of the event.
      ///
      /// \param[in] SimEvent
      ///     The name of the event.
      ///
      /// \return
      ///     The file position of the requested event.
      //-----------------------------------------------------------------
      positionSimEvent_type
      positionSimEvent( INT_4U Frame, const std::string& SimEvent ) const;

      //-----------------------------------------------------------------
      /// \brief Locate the position of the event
      ///
      /// \param[in] EventType
      ///     The name of the event.
      /// \param[in] Index
      ///     Offset of the event.
      ///
      /// \return
      ///     The file position of the requested event.
      //-----------------------------------------------------------------
      positionSimEvent_type
      positionSimEvent( const std::string& EventType, INT_4U Index ) const;

      //-----------------------------------------------------------------
      /// \brief Write the structure to the stream
      ///
      /// \param[in] Stream
      ///     The output stream where the object is to be written.
      //-----------------------------------------------------------------
      void write( Common::OStream& Stream ) const;
    };

    inline const FrTOCSimEvent::event_type& FrTOCSimEvent::event_type::
    operator=( const Previous::FrTOCSimEvent::event_type& Source )
    {
      GTime = Source.GTime;
      amplitudeSimEvent = Source.amplitudeSimEvent;
      positionSimEvent = Source.positionSimEvent;

      return *this;
    }

    inline Common::FrameSpec::size_type FrTOCSimEvent::
    Bytes( const Common::StreamBase& Stream ) const
    {
      Common::FrameSpec::size_type 
	retval = sizeof( nevent_type )
	+ sizeof( nTotalSEvent_type );
      //-----------------------------------------------------------------
      // 
      //-----------------------------------------------------------------
      for ( nameSimEvent_container_type::const_iterator
	      cur = m_info.begin( ),
	      last = m_info.end( );
	    cur != last;
	    ++cur )
      {
	retval +=
	  cur->first.Bytes( )
	  + sizeof( nevent_type )
	  + ( cur->second.size( ) * event_type_size )
	  ;
      }
      return retval;
    }

    template< typename SE >
    void FrTOCSimEvent::
    Description( Common::Description& Desc )
    {
      Desc( SE( "nSimEventType", "INT_4U",
		"Number of FrSimEvent in the file" ) );
      Desc( SE( "nameSimEvent", "STRING[nSimEventType]",
		"Array of FrSimEvent names" ) );
      Desc( SE( "nSimEvent", "INT_4U[nSimEventType]",
		"Number of FrSimEvent for each type of FrSimEvent"
		" (size of nSimEventType)" ) );
      Desc( SE( "nTotalSEvent", "INT_4U",
		"Total number of FrSimEvent" ) );
      Desc( SE( "GTimeSSim", "INT_4U[nTotalSEvent]",
		"GPS time in integer seconds" ) );
      Desc( SE( "GTimeNSim", "INT_4U[nTotalSEvent]",
		"Residual GPS time in integer nanoseconds" ) );
      Desc( SE( "amplitudeSimEvent", "REAL_4[nTotalSEvent]",
		"SimEvent amplitude" ) );
      Desc( SE( "positionSimEvent", "INT_8U[nTotalSEvent]",
		"Array of FrSimEvent positions, in bytes,"
		" from beginning of file" ) );
    }

    inline const FrTOCSimEvent::nameSimEvent_container_type& FrTOCSimEvent::
    GetSimEvent( ) const
    {
      return m_info;
    }

  } // namespace - Version_8
} // namespace - FrameCPP

#endif /* FrameCPP_VERSION_8_FrTOCSimEvent_HH */
