#define _POSIX_C_SOURCE 200809L
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include "koio_private.h"

static const int buckets_size = sizeof(((struct ko_hashtable *)0)->buckets) /
	sizeof(struct ko_hashtable_entry );

static unsigned int djb2(const char *str) {
	unsigned int hash = 5381;
	char c;
	while ((c = *str++)) {
		hash = ((hash << 5) + hash) + c;
	}
	return hash;
}

void *ko_hashtable_get(struct ko_hashtable *table, const char *key) {
	unsigned int hash = djb2(key);
	unsigned int bucket = hash % buckets_size;
	struct ko_hashtable_entry *entry = table->buckets[bucket];
	do {
		if (!entry || (entry->hash == hash &&
					strcmp(entry->key, key) == 0)) {
			break;
		}
		entry = entry->next;
	} while (entry);
	return entry ? entry->value : NULL;
}

void *ko_hashtable_set(struct ko_hashtable *table, const char *key, void *value) {
	unsigned int hash = djb2(key);
	unsigned int bucket = hash % buckets_size;
	struct ko_hashtable_entry *entry = table->buckets[bucket];
	struct ko_hashtable_entry *previous = NULL;

	do {
		if (!entry || (entry->hash == hash && strcmp(entry->key, key) == 0)) {
			break;
		}
		previous = entry;
		entry = entry->next;
	} while (entry);

	if (entry == NULL) {
		entry = calloc(1, sizeof(struct ko_hashtable_entry));
		entry->hash = hash;
		entry->key = strdup(key);
		if (previous) {
			previous->next = entry;
		} else {
			table->buckets[bucket] = entry;
		}
	}

	void *old = entry->value;
	entry->value = value;
	return old;
}

void *ko_hashtable_del(struct ko_hashtable *table, const void *key) {
	unsigned int hash = djb2(key);
	unsigned int bucket = hash % buckets_size;
	struct ko_hashtable_entry *entry = table->buckets[bucket];
	struct ko_hashtable_entry *previous = NULL;

	if (entry) {
		if (entry->hash != hash || strcmp(entry->key, key) != 0) {
			while (entry->next) {
				previous = entry;
				entry = entry->next;
				if (!entry || (entry->hash == hash &&
							strcmp(entry->key, key) == 0)) {
					break;
				}
			}
		}
	}

	if (entry == NULL) {
		return NULL;
	} else {
		void *old = entry->value;
		if (previous) {
			previous->next = entry->next;
		} else {
			table->buckets[bucket] = NULL;
		}
		free(entry->key);
		free(entry);
		return old;
	}
}
