/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { AuthType, TerminalQuotaError, ModelNotFoundError, PREVIEW_GEMINI_MODEL, } from '@google/gemini-cli-core';
import { useCallback, useEffect, useRef, useState } from 'react';
import {} from './useHistoryManager.js';
import { MessageType } from '../types.js';
import {} from '../contexts/UIStateContext.js';
export function useQuotaAndFallback({ config, historyManager, userTier, setModelSwitchedFromQuotaError, }) {
    const [proQuotaRequest, setProQuotaRequest] = useState(null);
    const isDialogPending = useRef(false);
    // Set up Flash fallback handler
    useEffect(() => {
        const fallbackHandler = async (failedModel, fallbackModel, error) => {
            // Fallbacks are currently only handled for OAuth users.
            const contentGeneratorConfig = config.getContentGeneratorConfig();
            if (!contentGeneratorConfig ||
                contentGeneratorConfig.authType !== AuthType.LOGIN_WITH_GOOGLE) {
                return null;
            }
            let message;
            let isTerminalQuotaError = false;
            let isModelNotFoundError = false;
            if (error instanceof TerminalQuotaError) {
                isTerminalQuotaError = true;
                // Common part of the message for both tiers
                const messageLines = [
                    `Usage limit reached for ${failedModel}.`,
                    error.retryDelayMs ? getResetTimeMessage(error.retryDelayMs) : null,
                    `/stats for usage details`,
                    `/auth to switch to API key.`,
                ].filter(Boolean);
                message = messageLines.join('\n');
            }
            else if (error instanceof ModelNotFoundError) {
                isModelNotFoundError = true;
                const messageLines = [
                    `It seems like you don't have access to Gemini 3.`,
                    `Learn more at https://goo.gle/enable-preview-features`,
                    `To disable Gemini 3, disable "Preview features" in /settings.`,
                ];
                message = messageLines.join('\n');
            }
            else {
                message = `${failedModel} is currently experiencing high demand. We apologize and appreciate your patience.`;
            }
            setModelSwitchedFromQuotaError(true);
            config.setQuotaErrorOccurred(true);
            if (isDialogPending.current) {
                return 'stop'; // A dialog is already active, so just stop this request.
            }
            isDialogPending.current = true;
            const intent = await new Promise((resolve) => {
                setProQuotaRequest({
                    failedModel,
                    fallbackModel,
                    resolve,
                    message,
                    isTerminalQuotaError,
                    isModelNotFoundError,
                });
            });
            return intent;
        };
        config.setFallbackModelHandler(fallbackHandler);
    }, [config, historyManager, userTier, setModelSwitchedFromQuotaError]);
    const handleProQuotaChoice = useCallback((choice) => {
        if (!proQuotaRequest)
            return;
        const intent = choice;
        proQuotaRequest.resolve(intent);
        setProQuotaRequest(null);
        isDialogPending.current = false; // Reset the flag here
        if (choice === 'retry_always') {
            // If we were recovering from a Preview Model failure, show a specific message.
            if (proQuotaRequest.failedModel === PREVIEW_GEMINI_MODEL) {
                historyManager.addItem({
                    type: MessageType.INFO,
                    text: `Switched to fallback model ${proQuotaRequest.fallbackModel}. ${!proQuotaRequest.isModelNotFoundError ? `We will periodically check if ${PREVIEW_GEMINI_MODEL} is available again.` : ''}`,
                }, Date.now());
            }
            else {
                historyManager.addItem({
                    type: MessageType.INFO,
                    text: 'Switched to fallback model.',
                }, Date.now());
            }
        }
    }, [proQuotaRequest, historyManager]);
    return {
        proQuotaRequest,
        handleProQuotaChoice,
    };
}
function getResetTimeMessage(delayMs) {
    const resetDate = new Date(Date.now() + delayMs);
    const timeFormatter = new Intl.DateTimeFormat('en-US', {
        hour: 'numeric',
        minute: '2-digit',
        timeZoneName: 'short',
    });
    return `Access resets at ${timeFormatter.format(resetDate)}.`;
}
//# sourceMappingURL=useQuotaAndFallback.js.map