
#include "comb/ordered-tree-lev-seq.h"
#include "comb/word-stats.h"


#include "fxtio.h"
#include "fxttypes.h"
#include "jjassert.h"

#include "nextarg.h"

//% Statistics for (level sequences of) ordered rooted trees.
//% Cf. the following OEIS sequences:
//% A000108: all trees.
//% A080936: trees by height.
//% A001263: trees by number of ascents.
//% A091894: trees by number of descents.
//% A033184: trees by number of ones.
//% A152879: trees by number of max levels.
//% A078391: trees by position of last 1.
//% A091869: trees by number of flat steps.
//% A116424: trees by number of peaks.
//% A203717: trees by max branching number (out-degree).


// Cf. comb/tree-lev-seq-stats-demo.cc for stats for unordered trees

//#define TIMING  // uncomment to disable printing


int
main(int argc, char **argv)
{
    ulong n = 5;
    NXARG(n, "Number of non-root nodes");

    ulong sq = 0;
    NXARG(sq, "Select stats:\n"
          "    0 ==> by height\n"
          "    1 ==> number of ascents\n"
          "    2 ==> number of descents\n"
          "    3 ==> number of ones\n"
//          "    4 ==> number of fixed points\n"  // same stats as 3
          "    5 ==> number of maximal levels\n"
          "    6 ==> position of last occurrence of one\n"
          "    7 ==> position of first occurrence of the maximal value\n"
          "    8 ==> position of last occurrence of the maximal value\n"
          "    9 ==> number of flat steps\n"
          "   10 ==> number of even values\n"
          "   11 ==> number of odd values\n"
          "   20 ==> number of flat steps\n"
          "   21 ==> number of non-flat steps\n"
          "   25 ==> number of peaks\n"
//          "   11 ==> number of \n"
          );

    bool aa = false;  // whether to print as ASCII art
    NXARG(aa,"Whether to render trees as ASCII art");

    ordered_tree_lev_seq T(n);
    const ulong N = T.size();  // == n + 1
    word_stats W(T.data(), N);

    ulong ct = 0;
    ulong * st = new ulong[N+1];  // stats
    for (ulong k=0; k < N+1; ++k)  st[k] = 0;


    ulong j = 0;
    do
    {
        ++ct;
        ulong s = 0;
        switch ( sq )
        {
        case 0:  // by height: A080936
            s = W.max_val();
            break;
            // 1,
            // 0, 1,
            // 0, 1, 1,
            // 0, 1, 3, 1,
            // 0, 1, 7, 5, 1,
            // 0, 1, 15, 18, 7, 1,
            // 0, 1, 31, 57, 33, 9, 1,
            // 0, 1, 63, 169, 132, 52, 11, 1,
            // 0, 1, 127, 482, 484, 247, 75, 13, 1,
            // 0, 1, 255, 1341, 1684, 1053, 410, 102, 15, 1,
            // 0, 1, 511, 3669, 5661, 4199, 1975, 629, 133, 17, 1,
            // 0, 1, 1023, 9922, 18579, 16017, 8778, 3366, 912, 168, 19, 1,

        case 1:  // by number of ascents: A001263
            s = W.num_ascents();
            break;
            // 1,
            // 0, 1,
            // 0, 1, 1,
            // 0, 1, 3, 1,
            // 0, 1, 6, 6, 1,
            // 0, 1, 10, 20, 10, 1,
            // 0, 1, 15, 50, 50, 15, 1,
            // 0, 1, 21, 105, 175, 105, 21, 1,
            // 0, 1, 28, 196, 490, 490, 196, 28, 1,
            // 0, 1, 36, 336, 1176, 1764, 1176, 336, 36, 1,
            // 0, 1, 45, 540, 2520, 5292, 5292, 2520, 540, 45, 1,
            // 0, 1, 55, 825, 4950, 13860, 19404, 13860, 4950, 825, 55, 1,


        case 2:  // by number of descents: A091894
            s = W.num_descents();
            break;
            // 1,
            // 1, 0,
            // 2, 0, 0,
            // 4, 1, 0, 0,
            // 8, 6, 0, 0, 0,
            // 16, 24, 2, 0, 0, 0,
            // 32, 80, 20, 0, 0, 0, 0,
            // 64, 240, 120, 5, 0, 0, 0, 0,
            // 128, 672, 560, 70, 0, 0, 0, 0, 0,
            // 256, 1792, 2240, 560, 14, 0, 0, 0, 0, 0,
            // 512, 4608, 8064, 3360, 252, 0, 0, 0, 0, 0, 0,
            // 1024, 11520, 26880, 16800, 2520, 42, 0, 0, 0, 0, 0, 0,


        case 3:  // by number of ones: A033184 (A009766 transposed)
            s = W.num_val(1);
            break;
            // 1,
            // 0, 1,
            // 0, 1, 1,
            // 0, 2, 2, 1,
            // 0, 5, 5, 3, 1,
            // 0, 14, 14, 9, 4, 1,
            // 0, 42, 42, 28, 14, 5, 1,
            // 0, 132, 132, 90, 48, 20, 6, 1,
            // 0, 429, 429, 297, 165, 75, 27, 7, 1,
            // 0, 1430, 1430, 1001, 572, 275, 110, 35, 8, 1,
            // 0, 4862, 4862, 3432, 2002, 1001, 429, 154, 44, 9, 1,
            // 0, 16796, 16796, 11934, 7072, 3640, 1638, 637, 208, 54, 10, 1,

        case 4:  // by number of fixed points: same as "number of ones"
            s = W.num_fixed_points();
            break;
            // 0, 1,
            // 0, 0, 1,
            // 0, 0, 1, 1,
            // 0, 0, 2, 2, 1,
            // 0, 0, 5, 5, 3, 1,
            // 0, 0, 14, 14, 9, 4, 1,
            // 0, 0, 42, 42, 28, 14, 5, 1,
            // 0, 0, 132, 132, 90, 48, 20, 6, 1,
            // 0, 0, 429, 429, 297, 165, 75, 27, 7, 1,
            // 0, 0, 1430, 1430, 1001, 572, 275, 110, 35, 8, 1,
            // 0, 0, 4862, 4862, 3432, 2002, 1001, 429, 154, 44, 9, 1,
            // 0, 0, 16796, 16796, 11934, 7072, 3640, 1638, 637, 208, 54, 10, 1,

        case 5:  // by number of max levels: A152879
            s = W.num_max_val();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 3, 1, 1, 0,
            // 0, 8, 4, 1, 1, 0,
            // 0, 23, 12, 5, 1, 1, 0,
            // 0, 71, 36, 17, 6, 1, 1, 0,
            // 0, 229, 114, 54, 23, 7, 1, 1, 0,
            // 0, 759, 377, 176, 78, 30, 8, 1, 1, 0,
            // 0, 2566, 1279, 596, 263, 109, 38, 9, 1, 1, 0,
            // 0, 8817, 4408, 2070, 912, 382, 148, 47, 10, 1, 1, 0,
            // 0, 30717, 15375, 7289, 3240, 1358, 541, 196, 57, 11, 1, 1, 0,

        case 6:  // position of last one: A078391
            s = W.last_val_idx(1);
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 2, 1, 2, 0,
            // 0, 5, 2, 2, 5, 0,
            // 0, 14, 5, 4, 5, 14, 0,
            // 0, 42, 14, 10, 10, 14, 42, 0,
            // 0, 132, 42, 28, 25, 28, 42, 132, 0,
            // 0, 429, 132, 84, 70, 70, 84, 132, 429, 0,
            // 0, 1430, 429, 264, 210, 196, 210, 264, 429, 1430, 0,
            // 0, 4862, 1430, 858, 660, 588, 588, 660, 858, 1430, 4862, 0,
            // 0, 16796, 4862, 2860, 2145, 1848, 1764, 1848, 2145, 2860, 4862, 16796, 0,


        case 7:  // position of first occurrence of max: A000000
            s = W.first_max_idx();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 2, 2, 0,
            // 0, 1, 4, 5, 4, 0,
            // 0, 1, 8, 12, 12, 9, 0,
            // 0, 1, 16, 29, 33, 31, 22, 0,
            // 0, 1, 32, 71, 90, 94, 84, 57, 0,
            // 0, 1, 64, 176, 247, 278, 274, 236, 154, 0,
            // 0, 1, 128, 441, 684, 820, 860, 816, 682, 430, 0,
            // 0, 1, 256, 1115, 1911, 2428, 2672, 2685, 2476, 2018, 1234, 0,
            // 0, 1, 512, 2840, 5382, 7231, 8296, 8690, 8474, 7641, 6094, 3625, 0,


        case 8:  // position of last occurrence of max: A000000
            s = W.last_max_idx();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 0, 2, 0,
            // 0, 0, 1, 4, 0,
            // 0, 0, 1, 4, 9, 0,
            // 0, 0, 1, 6, 13, 22, 0,
            // 0, 0, 1, 10, 24, 40, 57, 0,
            // 0, 0, 1, 18, 49, 85, 122, 154, 0,
            // 0, 0, 1, 34, 107, 196, 288, 374, 430, 0,
            // 0, 0, 1, 66, 244, 477, 723, 959, 1158, 1234, 0,
            // 0, 0, 1, 130, 573, 1204, 1893, 2568, 3176, 3626, 3625, 0,
            // 0, 0, 1, 258, 1375, 3121, 5108, 7101, 8956, 10520, 11481, 10865, 0,


        case 9:  // number of flat steps: A091869
            s = W.num_flat_steps();
            // column k=1: A000000
            break;
            // 1, 0,
            // 1, 0, 0,
            // 1, 1, 0, 0,
            // 2, 2, 1, 0, 0,
            // 4, 6, 3, 1, 0, 0,
            // 9, 16, 12, 4, 1, 0, 0,
            // 21, 45, 40, 20, 5, 1, 0, 0,
            // 51, 126, 135, 80, 30, 6, 1, 0, 0,
            // 127, 357, 441, 315, 140, 42, 7, 1, 0, 0,
            // 323, 1016, 1428, 1176, 630, 224, 56, 8, 1, 0, 0,
            // 835, 2907, 4572, 4284, 2646, 1134, 336, 72, 9, 1, 0, 0,
            // 2188, 8350, 14535, 15240, 10710, 5292, 1890, 480, 90, 10, 1, 0, 0,


        case 10:  // number of even values: A001263 (same as "number of ascents")
            s = W.num_even_val();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 3, 1, 0,
            // 0, 1, 6, 6, 1, 0,
            // 0, 1, 10, 20, 10, 1, 0,
            // 0, 1, 15, 50, 50, 15, 1, 0,
            // 0, 1, 21, 105, 175, 105, 21, 1, 0,
            // 0, 1, 28, 196, 490, 490, 196, 28, 1, 0,
            // 0, 1, 36, 336, 1176, 1764, 1176, 336, 36, 1, 0,
            // 0, 1, 45, 540, 2520, 5292, 5292, 2520, 540, 45, 1, 0,
            // 0, 1, 55, 825, 4950, 13860, 19404, 13860, 4950, 825, 55, 1, 0,

        case 11:  // number of odd values: (same as "number of even values")
            s = W.num_odd_val();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 3, 1, 0,
            // 0, 1, 6, 6, 1, 0,
            // 0, 1, 10, 20, 10, 1, 0,
            // 0, 1, 15, 50, 50, 15, 1, 0,
            // 0, 1, 21, 105, 175, 105, 21, 1, 0,
            // 0, 1, 28, 196, 490, 490, 196, 28, 1, 0,
            // 0, 1, 36, 336, 1176, 1764, 1176, 336, 36, 1, 0,
            // 0, 1, 45, 540, 2520, 5292, 5292, 2520, 540, 45, 1, 0,
            // 0, 1, 55, 825, 4950, 13860, 19404, 13860, 4950, 825, 55, 1, 0,

        case 20: // number of flat steps: A000000
            s = W.num_flat_steps();
            // column k=0: A000000
            break;
            // 1, 0,
            // 1, 0, 0,
            // 1, 1, 0, 0,
            // 2, 2, 1, 0, 0,
            // 4, 6, 3, 1, 0, 0,
            // 9, 16, 12, 4, 1, 0, 0,
            // 21, 45, 40, 20, 5, 1, 0, 0,
            // 51, 126, 135, 80, 30, 6, 1, 0, 0,
            // 127, 357, 441, 315, 140, 42, 7, 1, 0, 0,
            // 323, 1016, 1428, 1176, 630, 224, 56, 8, 1, 0, 0,
            // 835, 2907, 4572, 4284, 2646, 1134, 336, 72, 9, 1, 0, 0,
            // 2188, 8350, 14535, 15240, 10710, 5292, 1890, 480, 90, 10, 1, 0, 0,

        case 21: // number of non-flat steps: (same as "number of flat steps")
            s = W.num_nonflat_steps();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 2, 2, 0,
            // 0, 1, 3, 6, 4, 0,
            // 0, 1, 4, 12, 16, 9, 0,
            // 0, 1, 5, 20, 40, 45, 21, 0,
            // 0, 1, 6, 30, 80, 135, 126, 51, 0,
            // 0, 1, 7, 42, 140, 315, 441, 357, 127, 0,
            // 0, 1, 8, 56, 224, 630, 1176, 1428, 1016, 323, 0,
            // 0, 1, 9, 72, 336, 1134, 2646, 4284, 4572, 2907, 835, 0,
            // 0, 1, 10, 90, 480, 1890, 5292, 10710, 15240, 14535, 8350, 2188, 0,

        case 25: // number of peaks: A116424
            s = W.num_peaks();
            break;
            // 1, 0,
            // 1, 0, 0,
            // 2, 0, 0, 0,
            // 4, 1, 0, 0, 0,
            // 9, 5, 0, 0, 0, 0,
            // 22, 19, 1, 0, 0, 0, 0,
            // 57, 66, 9, 0, 0, 0, 0, 0,
            // 154, 221, 53, 1, 0, 0, 0, 0, 0,
            // 429, 729, 258, 14, 0, 0, 0, 0, 0, 0,
            // 1223, 2391, 1131, 116, 1, 0, 0, 0, 0, 0, 0,
            // 3550, 7829, 4652, 745, 20, 0, 0, 0, 0, 0, 0, 0,
            // 10455, 25638, 18357, 4115, 220, 1, 0, 0, 0, 0, 0, 0, 0,

        case 30: // max flat length: A000000
            s = W.max_flat_length();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 2, 2, 1, 0,
            // 0, 4, 7, 2, 1, 0,
            // 0, 9, 22, 8, 2, 1, 0,
            // 0, 21, 71, 29, 8, 2, 1, 0,
            // 0, 51, 232, 105, 30, 8, 2, 1, 0,
            // 0, 127, 766, 384, 112, 30, 8, 2, 1, 0,
            // 0, 323, 2552, 1413, 420, 113, 30, 8, 2, 1, 0,
            // 0, 835, 8572, 5223, 1585, 427, 113, 30, 8, 2, 1, 0,
            // 0, 2188, 29001, 19386, 6008, 1621, 428, 113, 30, 8, 2, 1, 0,

        case 40: // max frequency in histogram (of level seq.): A000000
            s = W.max_num_vals();
            break;
            // 0, 1,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 3, 1, 0,
            // 0, 1, 8, 4, 1, 0,
            // 0, 1, 18, 17, 5, 1, 0,
            // 0, 1, 41, 59, 24, 6, 1, 0,
            // 0, 1, 89, 190, 109, 32, 7, 1, 0,
            // 0, 1, 196, 591, 432, 160, 41, 8, 1, 0,
            // 0, 1, 424, 1787, 1617, 749, 223, 51, 9, 1, 0,
            // 0, 1, 925, 5260, 5905, 3202, 1131, 299, 62, 10, 1, 0,
            // 0, 1, 2003, 15380, 20853, 13041, 5402, 1631, 389, 74, 11, 1, 0,


        case 50:  // max branching number (out-degree): A203717
            s = T.max_branching_number();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 1, 3, 1, 0,
            // 0, 1, 8, 4, 1, 0,
            // 0, 1, 20, 15, 5, 1, 0,
            // 0, 1, 50, 53, 21, 6, 1, 0,
            // 0, 1, 126, 182, 84, 28, 7, 1, 0,
            // 0, 1, 322, 616, 326, 120, 36, 8, 1, 0,
            // 0, 1, 834, 2070, 1242, 495, 165, 45, 9, 1, 0,
            // 0, 1, 2187, 6930, 4680, 1997, 715, 220, 55, 10, 1, 0,
            // 0, 1, 5797, 23166, 17512, 7942, 3003, 1001, 286, 66, 11, 1, 0,

        case 51:  // min branching number (out-degree) of internal nodes: A000000
            s = T.min_branching_number();
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 1, 1, 0,
            // 0, 4, 0, 1, 0,
            // 0, 11, 2, 0, 1, 0,
            // 0, 36, 5, 0, 0, 1, 0,
            // 0, 117, 11, 3, 0, 0, 1, 0,
            // 0, 393, 28, 7, 0, 0, 0, 1, 0,
            // 0, 1339, 78, 8, 4, 0, 0, 0, 1, 0,
            // 0, 4630, 201, 21, 9, 0, 0, 0, 0, 1, 0,
            // 0, 16193, 532, 55, 10, 5, 0, 0, 0, 0, 1, 0,
            // 0, 57201, 1441, 121, 11, 11, 0, 0, 0, 0, 0, 1, 0,

        case 60:  // number of leaves: A108838
            s = T.num_leaves();
            // columns: k=1: A007290, k=2: A006470
            break;
            // 1, 0,
            // 0, 1, 0,
            // 0, 2, 0, 0,
            // 0, 3, 2, 0, 0,
            // 0, 4, 8, 2, 0, 0,
            // 0, 5, 20, 15, 2, 0, 0,
            // 0, 6, 40, 60, 24, 2, 0, 0,
            // 0, 7, 70, 175, 140, 35, 2, 0, 0,
            // 0, 8, 112, 420, 560, 280, 48, 2, 0, 0,
            // 0, 9, 168, 882, 1764, 1470, 504, 63, 2, 0, 0,
            // 0, 10, 240, 1680, 4704, 5880, 3360, 840, 80, 2, 0, 0,
            // 0, 11, 330, 2970, 11088, 19404, 16632, 6930, 1320, 99, 2, 0, 0,

        case 70:  // min limb length: A000000  (Dyck paths with shortest pyramid size k)
            s = T.min_limb_length();
            // columns: A000000, A000000
            break;
            // 1, 0,
            // 1, 0, 0,
            // 0, 1, 1, 0,
            // 0, 4, 0, 1, 0,
            // 0, 12, 1, 0, 1, 0,
            // 0, 38, 3, 0, 0, 1, 0,
            // 0, 124, 6, 1, 0, 0, 1, 0,
            // 0, 412, 13, 3, 0, 0, 0, 1, 0,
            // 0, 1393, 30, 5, 1, 0, 0, 0, 1, 0,
            // 0, 4780, 70, 8, 3, 0, 0, 0, 0, 1, 0,
            // 0, 16611, 163, 15, 5, 1, 0, 0, 0, 0, 1, 0,
            // 0, 58363, 381, 31, 7, 3, 0, 0, 0, 0, 0, 1, 0,
            // 0, 207034, 898, 63, 10, 5, 1, 0, 0, 0, 0, 0, 1, 0,
            // 0, 740617, 2131, 124, 17, 7, 3, 0, 0, 0, 0, 0, 0, 1, 0,

        case 71:  // max limb length: A120059 (Dyck paths with longest pyramid size k)
            s = T.max_limb_length();
            // columns: A086581
            break;
            // 1, 0,
            // 1, 0, 0,
            // 0, 1, 1, 0,
            // 0, 2, 2, 1, 0,
            // 0, 5, 6, 2, 1, 0,
            // 0, 13, 19, 7, 2, 1, 0,
            // 0, 35, 63, 24, 7, 2, 1, 0,
            // 0, 97, 212, 85, 25, 7, 2, 1, 0,
            // 0, 275, 723, 307, 90, 25, 7, 2, 1, 0,
            // 0, 794, 2491, 1121, 330, 91, 25, 7, 2, 1, 0,
            // 0, 2327, 8654, 4129, 1225, 335, 91, 25, 7, 2, 1, 0,
            // 0, 6905, 30273, 15313, 4585, 1248, 336, 91, 25, 7, 2, 1, 0,
            // 0, 20705, 106522, 57104, 17276, 4690, 1253, 336, 91, 25, 7, 2, 1, 0,
            // 0, 62642, 376727, 213915, 65446, 17741, 4713, 1254, 336, 91, 25, 7, 2, 1, 0,


        default:
            cerr << "Invalid choice (exiting)." << endl;
            return 1;
        }


        jjassert( s <= N );
        st[ s ] += 1;


#if !defined TIMING
        cout << setw(4) << ct << ":";

        // print RGS:
        T.print("  ");

        cout << setw(4) << s;

        if ( aa )
        {
            cout << endl;
            T.print_aa("");
            cout << endl;
        }

        cout << endl;

        //        jjassert( T.OK() );
#endif  // TIMING
    }
    while ( (j=T.next()) );


    ulong sct = 0;
    for (ulong k=0; k<=N; ++k)
    {
        cout << st[k] << ", ";
        sct += st[k];
    }
    cout << endl;

    cout << " ct=" << ct;  // total: OEIS sequence A000108
    cout << endl;

    jjassert( sct == ct );

    delete [] st;

    return 0;
}
// -------------------------

/*

  Print triangle for stats q:


  OEIS example:
  q=0; for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | nl -v0 -s':  ' -w2 -n rz

  C++ comment:
q=0;  for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | sed 's/^/\/\/ /; s/ $//;'

Extract column c from stats q:
q=0;  c=0;  echo $(for n in $(seq 0 11) ; do ./bin $n $q | grep ', $' ; done | cut -d, -f$((c+1))) | sed 's/ /, /g;'

*/


/// Emacs:
/// Local Variables:
/// MyRelDir: "demo/comb"
/// makefile-dir: "../../"
/// make-target: "1demo DSRC=demo/comb/ordered-tree-lev-seq-stats-demo.cc"
/// make-target2: "1demo DSRC=demo/comb/ordered-tree-lev-seq-stats-demo.cc DEMOFLAGS=-DTIMING"
/// End:
