% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/func_tapers.R
\name{tapers-constraints}
\alias{tapers-constraints}
\alias{constrain_tapers}
\alias{constrain_tapers.tapers}
\alias{constrain_tapers.default}
\alias{minspan}
\alias{minspan.tapers}
\alias{minspan.default}
\title{Taper constraint methods}
\usage{
constrain_tapers(tapvec, ...)

\method{constrain_tapers}{tapers}(tapvec, ...)

\method{constrain_tapers}{default}(
  tapvec,
  tapseq = NULL,
  constraint.method = c("simple.slope", "loess.smooth", "none"),
  verbose = TRUE,
  ...
)

minspan(tapvec, ...)

\method{minspan}{tapers}(tapvec, ...)

\method{minspan}{default}(tapvec, Kmin = NULL, Kmax = NULL, ...)
}
\arguments{
\item{tapvec}{integer or \code{'tapers'} object; the number of tapers at each frequency}

\item{...}{optional arguments}

\item{tapseq}{numeric; positions or frequencies -- necessary for smoother methods}

\item{constraint.method}{character; method to use for constraints on tapers numbers}

\item{verbose}{logical; should warnings and messages be given?}

\item{Kmin}{numeric; the minimum to set; default is 1}

\item{Kmax}{numeric; the maximum to set; default is the minimum of either (7/5 max value), or (1/2 series length)}
}
\value{
\code{\link{constrain_tapers}}: an object with class \code{'tapers'}; \code{\link{minspan}}: a vector
}
\description{
In the Riedel-Sidorenko recipe, the number of optimal tapers
at each frequency is strongly dependent on the first and
second derivatives of the spectrum. It is crucial to enforce
constraints on the number of actual tapers applied; this is
because the derivatives of "noisy" series can be bogus.

\code{\link{constrain_tapers}} refines the number of tapers at each frequency.

\code{\link{minspan}} sets bounds on the number of tapers at each frequency.
}
\details{
The method by which \code{\link{constrain_tapers}} refines tapers is 
set with the \code{constraint.method} argument:

\itemize{
  \item \code{'simple.slope'} use \code{\link{ctap_simple}}
  \item \code{'loess.smooth'} uses \code{\link{ctap_loess}}
  \item \code{'none'} returns unbounded tapers.
}

\code{\link{minspan}} bounds the number of tapers to within
the minimum of either the maximum number of tapers found in the object, 
or the half-length of the series, which is necessary because 
it would be nonsense to have more tapers than the length of the series. 

Details of the constraint methods:

\subsection{via first differencing (the default)}{

\code{\link{ctap_simple}} is the preferred constraint method.
The algorithm uses first-differencing to modify the number
of tapers in the previous position.  Effectively, the constraint
is based on a causal, 1st-order Finite Impulse-response Filter (FIR) 
which makes the method sensitive to rapid changes in the number of tapers; 
naturally, smoother spectra tend to produce less fluctuation in taper numbers, 
which makes this well suited for adaptive processing. 

This produces, generally, the most
stable results, meaning repeatedly running the constraint will not change values
other than on the first execution; the same cannot be said for the other
methods, which are also considerably more expensive to use.

}

\subsection{via LOESS smoothing}{

\code{\link{ctap_loess}} uses \code{\link{loess}} to smooth the taper vector; is
can be very slow thanks to quadratic scaling.

}
}
\section{Warning}{


\code{\link{ctap_loess}} results tend to be strongly dependent on
the tuning parameters given to \code{loess} (for obvious reasons); hence, 
some effort should be given to understand their effect, and/or re-tuning them if needed.
}

\examples{
\dontrun{#REX
library(psd)

##
## Taper constraint procedures
##

data(magnet)
X <- magnet$clean

##
## spectrum
PSD <- psdcore(X, ntaper=10, refresh=TRUE)
## optimize tapers
kopt <- riedsid(PSD)
kopt.loess  <- riedsid(PSD, c.method="loess.smooth")
# the preferred function:
kopt2 <- riedsid2(PSD)
#
plot(as.tapers(kopt2), ylim =c(0, 60))
lines(as.tapers(kopt.loess), col='black')
lines(as.tapers(kopt), col='black', lwd=2)

##
## To compare all the methods at once:
demo("ctap")

}#REX
}
\seealso{
\code{\link{riedsid}}, \code{\link{ctap_simple}}, \code{\link{ctap_loess}}, \code{\link{tapers}}
}
\author{
A.J. Barbour and R.L. Parker
}
