# Scaffolds

Scaffolds are basic structures that enable using webpack and npm in a consistent and robust manner. They also come with the advantage that no package which uses packer needs to rely on packer as a dependency: the package is meant to be used to develop others and should not figure in your `Imports`.

<Note type = "tip">
Most scaffolds can be used more than once per package, e.g.: to create multiple inputs.
</Note>

## Golem

The `scaffold_golem` function will lay down the scaffolding for a golem package.

```r
golem::create_golem("stocks") # create golem
packer::scaffold_golem()
```

```
── Scaffolding golem ──────────────────────────────────────────────────────

✔ Initialiased npm
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Added npm scripts
✔ Created srcjs directory
✔ Created path directory
✔ Created webpack config files

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/stocks'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

This will let you use packer to bundle files to `inst/app/www`, by default all files are bundled into one: `index.js`.

<Danger title="Unique" text="The scaffold for golem is unique in that only one can be laid down once." />

## Widgets

Htmlwidgets can also be scaffolded with `scaffold_widget`. Note that is will internally run `htmlwidgets::scaffoldWidget` so you should not run it prior to packer's scaffold.

```r
usethis::create_package("plotly")
packer::scaffold_widget("write_h1")
```

```
── Scaffolding widget ────────────────────────────────────────────── write_h1 ── 

✔ Bare widget setup
✔ Created path directory
✔ Initialiased npm
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Created path directory
✔ Created webpack config files
✔ Created path directory
✔ Created path directory
✔ Created srcjs/index.js
✔ Moved bare widget to srcjs
✔ Added npm scripts

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/plotly'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Adding packages to Imports ──

✔ Adding 'htmlwidgets' to Imports field in DESCRIPTION
● Refer to functions with `htmlwidgets::fun()`

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

This produces a toy widget which renders a message in HTML `h1` tags.

```r
packer::bundle()
devtools::document()
devtools::load_all()
write_h1("hello packer!")
```

![](_media/toy-widget.png)

<Danger title="Warning" text="Do not run scaffold the widget with the htmlwidgets package" />

## Extensions

In packer, a shiny extension is a package that extends much the way [shinyjs](https://deanattali.com/shinyjs/) or [waiter](https://waiter.john-coene.com/) does. Note that multiple extensions can be scaffolded within a single R package.

```r
usethis::create_package("alerts")
packer::scaffold_extension("ask")
```

```
── Scaffolding shiny extension ────────────────────────────────────────── ask ── 

✔ Initialiased npm
✔ Created path directory
✔ Created path directory
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Added npm scripts
✔ Created path directory
✔ Created webpack config files
✔ Created 'extension' module
✔ Created srcjs/index.js
✔ Added path to shiny resource
✔ Created R file and function

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/alerts'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Adding packages to Imports ──

✔ Adding 'shiny' to Imports field in DESCRIPTION
● Refer to functions with `shiny::fun()`

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

This builds a toy extension that uses vanilla JavaScript's `prompt` function to ask the user a question then relays the answer back to the server.

```r
packer::bundle()
devtools::document()
devtools::install()
```

```r
library(alerts)
library(shiny)

ui <- fluidPage(
  useAsk(),
  verbatimTextOutput("response")
)

server <- function(input, output){
ask("Please enter something:")
 output$response <- renderPrint({
   input$askResponse
 })
}

if(interactive())
 shinyApp(ui, server)
```

![_media](../_media/get-started.gif)

## Inputs

You can also use packer to scaffold custom shiny inputs.

```r
usethis::create_package("inputz")
packer::scaffold_input("increment")
```

```
── Scaffolding shiny input ──────────────────────────────────────── increment ── 

✔ Initialiased npm
✔ Created path directory
✔ Created path directory
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Created path directory
✔ Created webpack config files
✔ Created 'input' module
✔ Created srcjs/index.js
✔ Created R file and function
✔ Added npm scripts

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/inputz'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Adding packages to Imports ──

✔ Adding 'shiny' to Imports field in DESCRIPTION
● Refer to functions with `shiny::fun()`
✔ Adding 'htmltools' to Imports field in DESCRIPTION
● Refer to functions with `htmltools::fun()`

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

The toy example scaffolded by packer is a button that increments at every click.

```r
packer::bundle()
devtools::document()
devtools::install()
```

```r
library(inputz)
library(shiny)

ui <- fluidPage(
 incrementInput("theId", 0)
)

server <- function(input, output){

 observeEvent(input$theId, {
   print(input$theId)
 })

}

if(interactive())
 shinyApp(ui, server)
```

![](_media/input-basic.gif)

## Outputs

You can also scaffold a custom output for shiny.

```r
usethis::create_package("outputz")
packer::scaffold_output("markup")
```

```
── Scaffolding shiny output ────────────────────────────────────────── markup ── 

✔ Initialiased npm
✔ Created path directory
✔ Created path directory
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Added npm scripts
✔ Created path directory
✔ Created webpack config files
✔ Created 'output' module
✔ Created srcjs/index.js
✔ Created R file and function

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/outputz'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Adding packages to Imports ──

✔ Adding 'shiny' to Imports field in DESCRIPTION
● Refer to functions with `shiny::fun()`
✔ Adding 'htmltools' to Imports field in DESCRIPTION
● Refer to functions with `htmltools::fun()`

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

Then again, bundle the files and install the package to test it.

```r
packer::bundle()
devtools::document()
devtools::install()
```

Like other scaffolds the R code generated contains a toy example: it renders shiny tags in red.

```r
library(outputz)
library(shiny)

ui <- fluidPage(
 markupOutput("id")
)

server <- function(input, output){
 output$id <- renderMarkup({
   markup(h2("Hello"))
 })
}

if(interactive())
 shinyApp(ui, server)
```

![](_media/input-toy.png)

## R markdown

You can also scaffold for R markdown documents. This differs slightly from other scaffolds in that 1) only a single such scaffold can be set, 2) it is rooted in a project rather than a package. This is because structuring R markdown projects as packages is rather unwieldy; please open an issue if you know a better way.

Note that you can set either `react` or `vue` to `TRUE` when scaffolding.

```r
usethis::create_project("rmd")
packer::scaffold_rmd()
```

```
── Scaffolding rmd ───────────────────────────────────────────────────────────── 

✔ Initialiased npm
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Added npm scripts
✔ Created assets directory
✔ Created index.Rmd
✔ Created srcjs directory
✔ Created path directory
✔ Created webpack config files

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/Projects/rmd'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

This generated, amongst other things, an `index.Rmd` file which looks as below.

```yaml
---
title: "Packer"
output:
  html_document:
    includes:
      in_header: assets/header.html
      after_body: assets/app.html
---

# Packer
```

The `includes` contain what is necessary to include and run the JavaScript bundle in the HTML output, this will vary depending on the scaffold (Vanilla, React, or Vue).

## Ambiorix

You can also scaffold for an [ambiorix](https://ambiorix.john-coene.com) app, these must be run from the root of ambiorix projects created with the [ambiorix.generator](https://github.com/JohnCoene/ambiorix.generator).

```r
ambiorix.generator::create_basic("myapp")
packer::scaffold_ambiorix()
```

```
── Scaffolding ambiorix ────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────────── 
✔ Initialiased npm
✔ webpack, webpack-cli, webpack-merge installed with scope dev
✔ Added npm scripts
✔ Created srcjs directory
✔ Created srcjs/config directory
✔ Created webpack config files

── Adding files to .gitignore and .Rbuildignore ──

✔ Setting active project to '/myapp'
✔ Adding '^srcjs$' to '.Rbuildignore'
✔ Adding '^node_modules$' to '.Rbuildignore'
✔ Adding '^package\\.json$' to '.Rbuildignore'
✔ Adding '^package-lock\\.json$' to '.Rbuildignore'
✔ Adding '^webpack\\.dev\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.prod\\.js$' to '.Rbuildignore'
✔ Adding '^webpack\\.common\\.js$' to '.Rbuildignore'
✔ Adding 'node_modules' to '.gitignore'

── Scaffold built ──

ℹ Run `bundle` to build the JavaScript files
```

This can also be set for ambiorix to use Vue.
