% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{liferegr}
\alias{liferegr}
\title{Parametric regression models for failure time data}
\usage{
liferegr(
  data,
  rep = "rep",
  stratum = "stratum",
  time = "time",
  time2 = "time2",
  event = "event",
  covariates = "treat",
  weight = "weight",
  id = "id",
  dist = "weibull",
  robust = 0L
)
}
\arguments{
\item{data}{The input data frame that contains the following variables:
\itemize{
\item \code{rep}: The replication for by-group processing.
\item \code{stratum}: The stratum.
\item \code{time}: The follow-up time for right censored data, or
the left end of each interval for interval censored data.
\item \code{time2}: The right end of each interval for interval
censored data.
\item \code{event}: The event indicator, normally 1=event, 0=no event.
\item \code{covariates}: The values of baseline covariates.
This is the full-rank design matrix (excluding the intercept)
for the regression model, assuming that factor variables
have already been expanded into dummy variables.
The intercept will be added automatically.
\item \code{weight}: The weight for each observation.
\item \code{id}: The optional subject ID to group the score residuals
in computing the robust sandwich variance.
}}

\item{rep}{The name of the replication variable in the input data.}

\item{stratum}{The name of the stratum variable in the input data.}

\item{time}{The name of the time variable or the left end of each
interval for interval censored data in the input data.}

\item{time2}{The name of the right end of each interval for
interval censored data in the input data.}

\item{event}{The name of the event variable in the input data
for right censored data.}

\item{covariates}{The vector of names of baseline covariates
in the input data.}

\item{weight}{The name of the weighting variable in the input data.}

\item{id}{The name of the id variable in the input data.}

\item{dist}{The assumed distribution for time to event. Options include
"exponential", "weibull", "lognormal", and "loglogistic" to be
modeled on the log-scale, and "normal" and "logistic" to be modeled
on the original scale.}

\item{robust}{Whether a robust sandwich variance estimate should be
computed. The default is TRUE if there are fractional weights or
there is at least 1 id with >1 event. In the presence of the id
variable, the score residual will be aggregated for each id when
computing the robust sandwich variance estimate.}
}
\value{
A list with the following components:
\itemize{
\item \code{sumstat}: The data frame of summary statistics of model fit
with the following variables:
\itemize{
\item \code{rep}: The replication.
\item \code{n}: The number of observations.
\item \code{nevents}: The number of events.
\item \code{loglik0}: The log-likelihood under null.
\item \code{loglik1}: The maximum log-likelihood.
\item \code{scoretest}: The score test statistic.
}
\item \code{parest}: The data frame of parameter estimates with the
following variables:
\itemize{
\item \code{rep}: The replication.
\item \code{param}: The name of the covariate for the parameter estimate.
\item \code{beta}: The parameter estimate.
\item \code{sebeta}: The standard error of parameter estimate.
\item \code{z}: The Wald test statistic.
\item \code{expbeta}: The exponentiated parameter.
\item \code{vbeta}: The covariance matrix for parameter estimates.
}
}
}
\description{
Obtains the parameter estimates from parametric
regression models with uncensored, right censored, left censored, or
interval censored data.
}
\details{
There are two ways to specify the model, one for right censored
data through the time and event variables, and the other for interval
censored data through the time and time2 variables. For the second form,
we follow the convention used in SAS PROC LIFEREG:
\itemize{
\item If lower is not missing, upper is not missing, and lower is equal
to upper, then there is no censoring and the event occurred at
time lower.
\item If lower is not missing, upper is not missing, and lower < upper,
then the event time is censored within the interval (lower, upper).
\item If lower is missing, but upper is not missing, then upper will be
used as the left censoring value.
\item If lower is not missing, but upper is missing, then lower will be
used as the right censoring value.
\item If lower is not missing, upper is not missing, but lower > upper,
or if both lower and upper are missing, then the observation will
not be used.
}
}
\examples{

library(dplyr)

# right censored data
liferegr(data = rawdata \%>\% mutate(treat = 1*(treatmentGroup == 1)),
         rep = "iterationNumber", stratum = "stratum",
         time = "timeUnderObservation", event = "event",
         covariates = "treat", dist = "weibull")

# tobit regression for left censored data
liferegr(data = tobin \%>\% mutate(time = ifelse(durable>0, durable, NA)),
         time = "time", time2 = "durable",
         covariates = c("age", "quant"), dist = "normal")

}
\author{
Kaifeng Lu, \email{kaifenglu@gmail.com}
}
