% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/recm.R
\name{recm}
\alias{recm}
\title{Relational Evidential c-means algorithm}
\usage{
recm(
  D,
  c,
  type = "full",
  pairs = NULL,
  Omega = TRUE,
  m0 = NULL,
  ntrials = 1,
  alpha = 1,
  beta = 1.5,
  delta = quantile(D[upper.tri(D) | lower.tri(D)], 0.95),
  epsi = 1e-04,
  maxit = 5000,
  disp = TRUE
)
}
\arguments{
\item{D}{Dissimilarity matrix of size (n,n), where n is the number of objects.
Dissimilarities must be squared Euclidean distances to ensure convergence.}

\item{c}{Number of  clusters.}

\item{type}{Type of focal sets ("simple": empty set, singletons and Omega;
"full": all \eqn{2^c} subsets of \eqn{\Omega}; "pairs": \eqn{\emptyset}, singletons,
\eqn{\Omega}, and all or selected pairs).}

\item{pairs}{Set of pairs to be included in the focal sets; if NULL, all pairs are
included. Used only if type="pairs".}

\item{Omega}{Logical. If TRUE (default), the whole frame is included (for types 'simple' and
'pairs').}

\item{m0}{Initial credal partition. Should be a matrix with n rows and a number of
columns equal to the number f of focal sets specified by 'type' and 'pairs'.}

\item{ntrials}{Number of runs of the optimization algorithm (set to 1 if m0 is  supplied).}

\item{alpha}{Exponent of the cardinality in the cost function.}

\item{beta}{Exponent of masses in the cost function.}

\item{delta}{Distance to the empty set.}

\item{epsi}{Minimum amount of improvement.}

\item{maxit}{Maximum number of iterations.}

\item{disp}{If TRUE (default), intermediate results are displayed.}
}
\value{
The credal partition (an object of class \code{"credpart"}).
}
\description{
\code{recm} computes a credal partition from a dissimilarity matrix using the
Relational Evidential c-means (RECM) algorithm.
}
\details{
RECM is a relational version of the Evidential c-Means (ECM) algorithm. Convergence is
guaranteed only if elements of matrix D are squared Euclidean distances. However, the
algorithm is quite robust and generally provides sensible results even if the dissimilarities
are not metric. By default, each mass function in the credal partition has \eqn{2^c} focal sets,
where c is the supplied number of clusters. We can also limit the number of focal sets to
subsets of clusters with cardinalities 0, 1 and c (recommended if c>=10), or to all or some
selected pairs of clusters.
If an initial credal partition m0 is provided, the number of trials is automatically set to 1.
}
\examples{
## Clustering of the Butterfly dataset
\dontrun{
n <- nrow(butterfly)
D<-as.matrix(dist(butterfly))^2
clus<-recm(D,c=2,delta=sqrt(50))
m<-clus$mass
plot(1:n,m[,1],type="l",ylim=c(0,1),xlab="objects",ylab="masses")
lines(1:n,m[,2],lty=2)
lines(1:n,m[,3],lty=3)
lines(1:n,m[,4],lty=4)

 ## Clustering the protein data
 data(protein)
 clus <- recm(D=protein$D,c=4,type='full',alpha=0.2,beta=1.1,delta=sqrt(20))

 z<- cmdscale(protein$D,k=2)
 plot(clus,X=z,mfrow=c(2,2),ytrue=protein$y,Outliers=FALSE,Approx=1)
 }
}
\references{
M.-H. Masson and T. Denoeux. RECM: Relational Evidential c-means algorithm.
Pattern Recognition Letters, Vol. 30, pages 1015--1026, 2009.
}
\seealso{
\code{\link{makeF}}, \code{\link{extractMass}}, \code{\link{ecm}}
}
\author{
Thierry Denoeux (from a MATLAB code written by Marie-Helene Masson).
}
