% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model.R
\name{model-method-variational}
\alias{model-method-variational}
\alias{variational}
\title{Run Stan's variational approximation algorithms}
\usage{
variational(
  data = NULL,
  seed = NULL,
  refresh = NULL,
  init = NULL,
  save_latent_dynamics = FALSE,
  output_dir = NULL,
  output_basename = NULL,
  sig_figs = NULL,
  threads = NULL,
  opencl_ids = NULL,
  algorithm = NULL,
  iter = NULL,
  grad_samples = NULL,
  elbo_samples = NULL,
  eta = NULL,
  adapt_engaged = NULL,
  adapt_iter = NULL,
  tol_rel_obj = NULL,
  eval_elbo = NULL,
  output_samples = NULL,
  draws = NULL
)
}
\arguments{
\item{data}{(multiple options) The data to use for the variables specified in
the data block of the Stan program. One of the following:
\itemize{
\item A named list of \R objects with the names corresponding to variables
declared in the data block of the Stan program. Internally this list is then
written to JSON for CmdStan using \code{\link[=write_stan_json]{write_stan_json()}}. See
\code{\link[=write_stan_json]{write_stan_json()}} for details on the conversions performed on \R objects
before they are passed to Stan.
\item A path to a data file compatible with CmdStan (JSON or \R dump). See the
appendices in the CmdStan guide for details on using these formats.
\item \code{NULL} or an empty list if the Stan program has no data block.
}}

\item{seed}{(positive integer(s)) A seed for the (P)RNG to pass to CmdStan.
In the case of multi-chain sampling the single \code{seed} will automatically be
augmented by the the run (chain) ID so that each chain uses a different
seed. The exception is the transformed data block, which defaults to
using same seed for all chains so that the same data is generated for all
chains if RNG functions are used. The only time \code{seed} should be specified
as a vector (one element per chain) is if RNG functions are used in
transformed data and the goal is to generate \emph{different} data for each
chain.}

\item{refresh}{(non-negative integer) The number of iterations between
printed screen updates. If \code{refresh = 0}, only error messages will be
printed.}

\item{init}{(multiple options) The initialization method to use for the
variables declared in the parameters block of the Stan program. One of
the following:
\itemize{
\item A real number \code{x>0}. This initializes \emph{all} parameters randomly between
\verb{[-x,x]} on the \emph{unconstrained} parameter space.;
\item The number \code{0}. This initializes \emph{all} parameters to \code{0};
\item A character vector of paths (one per chain) to JSON or Rdump files
containing initial values for all or some parameters. See
\code{\link[=write_stan_json]{write_stan_json()}} to write \R objects to JSON files compatible with
CmdStan.
\item A list of lists containing initial values for all or some parameters. For
MCMC the list should contain a sublist for each chain. For other model
fitting methods there should be just one sublist. The sublists should have
named elements corresponding to the parameters for which you are specifying
initial values. See \strong{Examples}.
\item A function that returns a single list with names corresponding to the
parameters for which you are specifying initial values. The function can
take no arguments or a single argument \code{chain_id}. For MCMC, if the function
has argument \code{chain_id} it will be supplied with the chain id (from 1 to
number of chains) when called to generate the initial values. See
\strong{Examples}.
}}

\item{save_latent_dynamics}{(logical) Should auxiliary diagnostic information
about the latent dynamics be written to temporary diagnostic CSV files?
This argument replaces CmdStan's \code{diagnostic_file} argument and the content
written to CSV is controlled by the user's CmdStan installation and not
CmdStanR (for some algorithms no content may be written). The default
is \code{FALSE}, which is appropriate for almost every use case. To save the
temporary files created when \code{save_latent_dynamics=TRUE} see the
\code{\link[=fit-method-save_latent_dynamics_files]{$save_latent_dynamics_files()}}
method.}

\item{output_dir}{(string) A path to a directory where CmdStan should write
its output CSV files. For interactive use this can typically be left at
\code{NULL} (temporary directory) since CmdStanR makes the CmdStan output
(posterior draws and diagnostics) available in \R via methods of the fitted
model objects. The behavior of \code{output_dir} is as follows:
\itemize{
\item If \code{NULL} (the default), then the CSV files are written to a temporary
directory and only saved permanently if the user calls one of the \verb{$save_*}
methods of the fitted model object (e.g.,
\code{\link[=fit-method-save_output_files]{$save_output_files()}}). These temporary
files are removed when the fitted model object is
\link[base:gc]{garbage collected} (manually or automatically).
\item If a path, then the files are created in \code{output_dir} with names
corresponding to the defaults used by \verb{$save_output_files()}.
}}

\item{output_basename}{(string) A string to use as a prefix for the names of
the output CSV files of CmdStan. If \code{NULL} (the default), the basename of
the output CSV files will be comprised from the model name, timestamp, and
5 random characters.}

\item{sig_figs}{(positive integer) The number of significant figures used
when storing the output values. By default, CmdStan represent the output
values with 6 significant figures. The upper limit for \code{sig_figs} is 18.
Increasing this value will result in larger output CSV files and thus an
increased usage of disk space.}

\item{threads}{(positive integer) If the model was
\link[=model-method-compile]{compiled} with threading support, the number of
threads to use in parallelized sections (e.g., when using the Stan
functions \code{reduce_sum()} or \code{map_rect()}).}

\item{opencl_ids}{(integer vector of length 2) The platform and
device IDs of the OpenCL device to use for fitting. The model must
be compiled with \code{cpp_options = list(stan_opencl = TRUE)} for this
argument to have an effect.}

\item{algorithm}{(string) The algorithm. Either \code{"meanfield"} or
\code{"fullrank"}.}

\item{iter}{(positive integer) The \emph{maximum} number of iterations.}

\item{grad_samples}{(positive integer) The number of samples for Monte Carlo
estimate of gradients.}

\item{elbo_samples}{(positive integer) The number of samples for Monte Carlo
estimate of ELBO (objective function).}

\item{eta}{(positive real) The step size weighting parameter for adaptive
step size sequence.}

\item{adapt_engaged}{(logical) Do warmup adaptation?}

\item{adapt_iter}{(positive integer) The \emph{maximum} number of adaptation
iterations.}

\item{tol_rel_obj}{(positive real) Convergence tolerance on the relative norm
of the objective.}

\item{eval_elbo}{(positive integer) Evaluate ELBO every Nth iteration.}

\item{output_samples}{(positive integer) Use \code{draws} argument instead.
\code{output_samples} will be deprecated in the future.}

\item{draws}{(positive integer) Number of approximate posterior
samples to draw and save.}
}
\value{
A \code{\link{CmdStanVB}} object.
}
\description{
The \verb{$variational()} method of a \code{\link{CmdStanModel}} object runs
Stan's Automatic Differentiation Variational Inference (ADVI) algorithms.
The approximation is a Gaussian in the unconstrained variable space. Stan
implements two ADVI algorithms: the \code{algorithm="meanfield"} option uses a
fully factorized Gaussian for the approximation; the \code{algorithm="fullrank"}
option uses a Gaussian with a full-rank covariance matrix for the
approximation. See the
\href{https://mc-stan.org/docs/cmdstan-guide/}{CmdStan User’s Guide}
for more details.

Any argument left as \code{NULL} will default to the default value used by the
installed version of CmdStan.
}
\examples{
\dontrun{
library(cmdstanr)
library(posterior)
library(bayesplot)
color_scheme_set("brightblue")

# Set path to CmdStan
# (Note: if you installed CmdStan via install_cmdstan() with default settings
# then setting the path is unnecessary but the default below should still work.
# Otherwise use the `path` argument to specify the location of your
# CmdStan installation.)
set_cmdstan_path(path = NULL)

# Create a CmdStanModel object from a Stan program,
# here using the example model that comes with CmdStan
file <- file.path(cmdstan_path(), "examples/bernoulli/bernoulli.stan")
mod <- cmdstan_model(file)
mod$print()

# Data as a named list (like RStan)
stan_data <- list(N = 10, y = c(0,1,0,0,0,0,0,0,0,1))

# Run MCMC using the 'sample' method
fit_mcmc <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  parallel_chains = 2
)

# Use 'posterior' package for summaries
fit_mcmc$summary()

# Check sampling diagnostics
fit_mcmc$diagnostic_summary()

# Get posterior draws
draws <- fit_mcmc$draws()
print(draws)

# Convert to data frame using posterior::as_draws_df
as_draws_df(draws)

# Plot posterior using bayesplot (ggplot2)
mcmc_hist(fit_mcmc$draws("theta"))

# For models fit using MCMC, if you like working with RStan's stanfit objects
# then you can create one with rstan::read_stan_csv()
# stanfit <- rstan::read_stan_csv(fit_mcmc$output_files())


# Run 'optimize' method to get a point estimate (default is Stan's LBFGS algorithm)
# and also demonstrate specifying data as a path to a file instead of a list
my_data_file <- file.path(cmdstan_path(), "examples/bernoulli/bernoulli.data.json")
fit_optim <- mod$optimize(data = my_data_file, seed = 123)
fit_optim$summary()

# Run 'optimize' again with 'jacobian=TRUE' and then draw from Laplace approximation
# to the posterior
fit_optim <- mod$optimize(data = my_data_file, jacobian = TRUE)
fit_laplace <- mod$laplace(data = my_data_file, mode = fit_optim, draws = 2000)
fit_laplace$summary()

# Run 'variational' method to use ADVI to approximate posterior
fit_vb <- mod$variational(data = stan_data, seed = 123)
fit_vb$summary()
mcmc_hist(fit_vb$draws("theta"))

# Run 'pathfinder' method, a new alternative to the variational method
fit_pf <- mod$pathfinder(data = stan_data, seed = 123)
fit_pf$summary()
mcmc_hist(fit_pf$draws("theta"))

# Run 'pathfinder' again with more paths, fewer draws per path,
# better covariance approximation, and fewer LBFGSs iterations
fit_pf <- mod$pathfinder(data = stan_data, num_paths=10, single_path_draws=40,
                         history_size=50, max_lbfgs_iters=100)

# Specifying initial values as a function
fit_mcmc_w_init_fun <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = function() list(theta = runif(1))
)
fit_mcmc_w_init_fun_2 <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = function(chain_id) {
    # silly but demonstrates optional use of chain_id
    list(theta = 1 / (chain_id + 1))
  }
)
fit_mcmc_w_init_fun_2$init()

# Specifying initial values as a list of lists
fit_mcmc_w_init_list <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = list(
    list(theta = 0.75), # chain 1
    list(theta = 0.25)  # chain 2
  )
)
fit_optim_w_init_list <- mod$optimize(
  data = stan_data,
  seed = 123,
  init = list(
    list(theta = 0.75)
  )
)
fit_optim_w_init_list$init()
}

}
\seealso{
The CmdStanR website
(\href{https://mc-stan.org/cmdstanr/}{mc-stan.org/cmdstanr}) for online
documentation and tutorials.

The Stan and CmdStan documentation:
\itemize{
\item Stan documentation: \href{https://mc-stan.org/users/documentation/}{mc-stan.org/users/documentation}
\item CmdStan User’s Guide: \href{https://mc-stan.org/docs/cmdstan-guide/}{mc-stan.org/docs/cmdstan-guide}
}

Other CmdStanModel methods: 
\code{\link{model-method-check_syntax}},
\code{\link{model-method-compile}},
\code{\link{model-method-diagnose}},
\code{\link{model-method-expose_functions}},
\code{\link{model-method-format}},
\code{\link{model-method-generate-quantities}},
\code{\link{model-method-laplace}},
\code{\link{model-method-optimize}},
\code{\link{model-method-pathfinder}},
\code{\link{model-method-sample_mpi}},
\code{\link{model-method-sample}},
\code{\link{model-method-variables}}
}
\concept{CmdStanModel methods}
