% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model.R
\name{cmdstan_model}
\alias{cmdstan_model}
\title{Create a new CmdStanModel object}
\usage{
cmdstan_model(stan_file = NULL, exe_file = NULL, compile = TRUE, ...)
}
\arguments{
\item{stan_file}{(string) The path to a \code{.stan} file containing a Stan
program. The helper function \code{\link[=write_stan_file]{write_stan_file()}} is provided for cases when
it is more convenient to specify the Stan program as a string. If
\code{stan_file} is not specified then \code{exe_file} must be specified.}

\item{exe_file}{(string) The path to an existing Stan model executable. Can
be provided instead of or in addition to \code{stan_file} (if \code{stan_file} is
omitted some \code{CmdStanModel} methods like \verb{$code()} and \verb{$print()} will not
work). This argument can only be used with CmdStan 2.27+.}

\item{compile}{(logical) Do compilation? The default is \code{TRUE}. If \code{FALSE}
compilation can be done later via the \code{\link[=model-method-compile]{$compile()}}
method.}

\item{...}{Optionally, additional arguments to pass to the
\code{\link[=model-method-compile]{$compile()}} method if \code{compile=TRUE}. These
options include specifying the directory for saving the executable, turning
on pedantic mode, specifying include paths, configuring C++ options, and
more. See \code{\link[=model-method-compile]{$compile()}} for details.}
}
\value{
A \code{\link{CmdStanModel}} object.
}
\description{
\if{html}{\figure{logo.png}{options: width=25}}
Create a new \code{\link{CmdStanModel}} object from a file containing a Stan program
or from an existing Stan executable. The \code{\link{CmdStanModel}} object stores the
path to a Stan program and compiled executable (once created), and provides
methods for fitting the model using Stan's algorithms.

See the \code{compile} and \code{...} arguments for control over whether and how
compilation happens.
}
\examples{
\dontrun{
library(cmdstanr)
library(posterior)
library(bayesplot)
color_scheme_set("brightblue")

# Set path to CmdStan
# (Note: if you installed CmdStan via install_cmdstan() with default settings
# then setting the path is unnecessary but the default below should still work.
# Otherwise use the `path` argument to specify the location of your
# CmdStan installation.)
set_cmdstan_path(path = NULL)

# Create a CmdStanModel object from a Stan program,
# here using the example model that comes with CmdStan
file <- file.path(cmdstan_path(), "examples/bernoulli/bernoulli.stan")
mod <- cmdstan_model(file)
mod$print()

# Data as a named list (like RStan)
stan_data <- list(N = 10, y = c(0,1,0,0,0,0,0,0,0,1))

# Run MCMC using the 'sample' method
fit_mcmc <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  parallel_chains = 2
)

# Use 'posterior' package for summaries
fit_mcmc$summary()

# Check sampling diagnostics
fit_mcmc$diagnostic_summary()

# Get posterior draws
draws <- fit_mcmc$draws()
print(draws)

# Convert to data frame using posterior::as_draws_df
as_draws_df(draws)

# Plot posterior using bayesplot (ggplot2)
mcmc_hist(fit_mcmc$draws("theta"))

# For models fit using MCMC, if you like working with RStan's stanfit objects
# then you can create one with rstan::read_stan_csv()
# stanfit <- rstan::read_stan_csv(fit_mcmc$output_files())


# Run 'optimize' method to get a point estimate (default is Stan's LBFGS algorithm)
# and also demonstrate specifying data as a path to a file instead of a list
my_data_file <- file.path(cmdstan_path(), "examples/bernoulli/bernoulli.data.json")
fit_optim <- mod$optimize(data = my_data_file, seed = 123)
fit_optim$summary()

# Run 'optimize' again with 'jacobian=TRUE' and then draw from Laplace approximation
# to the posterior
fit_optim <- mod$optimize(data = my_data_file, jacobian = TRUE)
fit_laplace <- mod$laplace(data = my_data_file, mode = fit_optim, draws = 2000)
fit_laplace$summary()

# Run 'variational' method to use ADVI to approximate posterior
fit_vb <- mod$variational(data = stan_data, seed = 123)
fit_vb$summary()
mcmc_hist(fit_vb$draws("theta"))

# Run 'pathfinder' method, a new alternative to the variational method
fit_pf <- mod$pathfinder(data = stan_data, seed = 123)
fit_pf$summary()
mcmc_hist(fit_pf$draws("theta"))

# Run 'pathfinder' again with more paths, fewer draws per path,
# better covariance approximation, and fewer LBFGSs iterations
fit_pf <- mod$pathfinder(data = stan_data, num_paths=10, single_path_draws=40,
                         history_size=50, max_lbfgs_iters=100)

# Specifying initial values as a function
fit_mcmc_w_init_fun <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = function() list(theta = runif(1))
)
fit_mcmc_w_init_fun_2 <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = function(chain_id) {
    # silly but demonstrates optional use of chain_id
    list(theta = 1 / (chain_id + 1))
  }
)
fit_mcmc_w_init_fun_2$init()

# Specifying initial values as a list of lists
fit_mcmc_w_init_list <- mod$sample(
  data = stan_data,
  seed = 123,
  chains = 2,
  refresh = 0,
  init = list(
    list(theta = 0.75), # chain 1
    list(theta = 0.25)  # chain 2
  )
)
fit_optim_w_init_list <- mod$optimize(
  data = stan_data,
  seed = 123,
  init = list(
    list(theta = 0.75)
  )
)
fit_optim_w_init_list$init()
}

}
\seealso{
\code{\link[=install_cmdstan]{install_cmdstan()}}, \code{\link[=model-method-compile]{$compile()}},
\code{\link[=model-method-check_syntax]{$check_syntax()}}

The CmdStanR website
(\href{https://mc-stan.org/cmdstanr/}{mc-stan.org/cmdstanr}) for online
documentation and tutorials.

The Stan and CmdStan documentation:
\itemize{
\item Stan documentation: \href{https://mc-stan.org/users/documentation/}{mc-stan.org/users/documentation}
\item CmdStan User’s Guide: \href{https://mc-stan.org/docs/cmdstan-guide/}{mc-stan.org/docs/cmdstan-guide}
}
}
