% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model.R
\name{model-method-generate-quantities}
\alias{model-method-generate-quantities}
\alias{generate_quantities}
\title{Run Stan's standalone generated quantities method}
\usage{
generate_quantities(
  fitted_params,
  data = NULL,
  seed = NULL,
  output_dir = NULL,
  output_basename = NULL,
  sig_figs = NULL,
  parallel_chains = getOption("mc.cores", 1),
  threads_per_chain = NULL,
  opencl_ids = NULL
)
}
\arguments{
\item{fitted_params}{(multiple options) The parameter draws to use. One of
the following:
\itemize{
\item A \link{CmdStanMCMC} or \link{CmdStanVB} fitted model object.
\item A \link[posterior:draws_array]{posterior::draws_array} (for MCMC) or \link[posterior:draws_matrix]{posterior::draws_matrix} (for
VB) object returned by CmdStanR's \code{\link[=fit-method-draws]{$draws()}} method.
\item A character vector of paths to CmdStan CSV output files.
}

NOTE: if you plan on making many calls to \verb{$generate_quantities()} then the
most efficient option is to pass the paths of the CmdStan CSV output files
(this avoids CmdStanR having to rewrite the draws contained in the fitted
model object to CSV each time). If you no longer have the CSV files you can
use \code{\link[=draws_to_csv]{draws_to_csv()}} once to write them and then pass the resulting file
paths to \verb{$generate_quantities()} as many times as needed.}

\item{data}{(multiple options) The data to use for the variables specified in
the data block of the Stan program. One of the following:
\itemize{
\item A named list of \R objects with the names corresponding to variables
declared in the data block of the Stan program. Internally this list is then
written to JSON for CmdStan using \code{\link[=write_stan_json]{write_stan_json()}}. See
\code{\link[=write_stan_json]{write_stan_json()}} for details on the conversions performed on \R objects
before they are passed to Stan.
\item A path to a data file compatible with CmdStan (JSON or \R dump). See the
appendices in the CmdStan guide for details on using these formats.
\item \code{NULL} or an empty list if the Stan program has no data block.
}}

\item{seed}{(positive integer(s)) A seed for the (P)RNG to pass to CmdStan.
In the case of multi-chain sampling the single \code{seed} will automatically be
augmented by the the run (chain) ID so that each chain uses a different
seed. The exception is the transformed data block, which defaults to
using same seed for all chains so that the same data is generated for all
chains if RNG functions are used. The only time \code{seed} should be specified
as a vector (one element per chain) is if RNG functions are used in
transformed data and the goal is to generate \emph{different} data for each
chain.}

\item{output_dir}{(string) A path to a directory where CmdStan should write
its output CSV files. For interactive use this can typically be left at
\code{NULL} (temporary directory) since CmdStanR makes the CmdStan output
(posterior draws and diagnostics) available in \R via methods of the fitted
model objects. The behavior of \code{output_dir} is as follows:
\itemize{
\item If \code{NULL} (the default), then the CSV files are written to a temporary
directory and only saved permanently if the user calls one of the \verb{$save_*}
methods of the fitted model object (e.g.,
\code{\link[=fit-method-save_output_files]{$save_output_files()}}). These temporary
files are removed when the fitted model object is
\link[base:gc]{garbage collected} (manually or automatically).
\item If a path, then the files are created in \code{output_dir} with names
corresponding to the defaults used by \verb{$save_output_files()}.
}}

\item{output_basename}{(string) A string to use as a prefix for the names of
the output CSV files of CmdStan. If \code{NULL} (the default), the basename of
the output CSV files will be comprised from the model name, timestamp, and
5 random characters.}

\item{sig_figs}{(positive integer) The number of significant figures used
when storing the output values. By default, CmdStan represent the output
values with 6 significant figures. The upper limit for \code{sig_figs} is 18.
Increasing this value will result in larger output CSV files and thus an
increased usage of disk space.}

\item{parallel_chains}{(positive integer) The \emph{maximum} number of MCMC chains
to run in parallel. If \code{parallel_chains} is not specified then the default
is to look for the option \code{"mc.cores"}, which can be set for an entire \R
session by \code{options(mc.cores=value)}. If the \code{"mc.cores"} option has not
been set then the default is \code{1}.}

\item{threads_per_chain}{(positive integer) If the model was
\link[=model-method-compile]{compiled} with threading support, the number of
threads to use in parallelized sections \emph{within} an MCMC chain (e.g., when
using the Stan functions \code{reduce_sum()} or \code{map_rect()}). This is in
contrast with \code{parallel_chains}, which specifies the number of chains to
run in parallel. The actual number of CPU cores used is
\code{parallel_chains*threads_per_chain}. For an example of using threading see
the Stan case study
\href{https://mc-stan.org/users/documentation/case-studies/reduce_sum_tutorial.html}{Reduce Sum: A Minimal Example}.}

\item{opencl_ids}{(integer vector of length 2) The platform and
device IDs of the OpenCL device to use for fitting. The model must
be compiled with \code{cpp_options = list(stan_opencl = TRUE)} for this
argument to have an effect.}
}
\value{
A \code{\link{CmdStanGQ}} object.
}
\description{
The \verb{$generate_quantities()} method of a \code{\link{CmdStanModel}} object
runs Stan's standalone generated quantities to obtain generated quantities
based on previously fitted parameters.
}
\examples{
\dontrun{
# first fit a model using MCMC
mcmc_program <- write_stan_file(
  "data {
    int<lower=0> N;
    int<lower=0,upper=1> y[N];
  }
  parameters {
    real<lower=0,upper=1> theta;
  }
  model {
    y ~ bernoulli(theta);
  }"
)
mod_mcmc <- cmdstan_model(mcmc_program)

data <- list(N = 10, y = c(1,1,0,0,0,1,0,1,0,0))
fit_mcmc <- mod_mcmc$sample(data = data, seed = 123, refresh = 0)

# stan program for standalone generated quantities
# (could keep model block, but not necessary so removing it)
gq_program <- write_stan_file(
  "data {
    int<lower=0> N;
    int<lower=0,upper=1> y[N];
  }
  parameters {
    real<lower=0,upper=1> theta;
  }
  generated quantities {
    int y_rep[N] = bernoulli_rng(rep_vector(theta, N));
  }"
)

mod_gq <- cmdstan_model(gq_program)
fit_gq <- mod_gq$generate_quantities(fit_mcmc, data = data, seed = 123)
str(fit_gq$draws())

library(posterior)
as_draws_df(fit_gq$draws())
}

}
\seealso{
The CmdStanR website
(\href{https://mc-stan.org/cmdstanr/}{mc-stan.org/cmdstanr}) for online
documentation and tutorials.

The Stan and CmdStan documentation:
\itemize{
\item Stan documentation: \href{https://mc-stan.org/users/documentation/}{mc-stan.org/users/documentation}
\item CmdStan User’s Guide: \href{https://mc-stan.org/docs/cmdstan-guide/}{mc-stan.org/docs/cmdstan-guide}
}

Other CmdStanModel methods: 
\code{\link{model-method-check_syntax}},
\code{\link{model-method-compile}},
\code{\link{model-method-diagnose}},
\code{\link{model-method-format}},
\code{\link{model-method-optimize}},
\code{\link{model-method-sample_mpi}},
\code{\link{model-method-sample}},
\code{\link{model-method-variables}},
\code{\link{model-method-variational}}
}
\concept{CmdStanModel methods}
