## ----eval = FALSE-------------------------------------------------------------
#  if (!require("BiocManager"))
#      install.packages("BiocManager")
#  BiocManager::install("RaggedExperiment")

## ----include=TRUE,results="hide",message=FALSE,warning=FALSE------------------
library(RaggedExperiment)
library(GenomicRanges)

## ----echo = FALSE, fig.cap = "RaggedExperiment object schematic. Rows and columns represent genomic ranges and samples, respectively. Assay operations can be performed with (from left to right) compactAssay, qreduceAssay, and sparseAssay.", out.width = "\\maxwidth"----
knitr::include_graphics("RaggedExperiment.svg")

## -----------------------------------------------------------------------------
sample1 <- GRanges(
    c(A = "chr1:1-10:-", B = "chr1:8-14:+", C = "chr2:15-18:+"),
    score = 3:5)
sample2 <- GRanges(
    c(D = "chr1:1-10:-", E = "chr2:11-18:+"),
    score = 1:2)

## -----------------------------------------------------------------------------
colDat <- DataFrame(id = 1:2)

## -----------------------------------------------------------------------------
ragexp <- RaggedExperiment(
    sample1 = sample1,
    sample2 = sample2,
    colData = colDat
)
ragexp

## -----------------------------------------------------------------------------
grl <- GRangesList(sample1 = sample1, sample2 = sample2)
RaggedExperiment(grl, colData = colDat)

## -----------------------------------------------------------------------------
rangeList <- list(sample1 = sample1, sample2 = sample2)
RaggedExperiment(rangeList, colData = colDat)

## -----------------------------------------------------------------------------
grList <- List(sample1 = sample1, sample2 = sample2)
mcols(grList) <- colDat
RaggedExperiment(grList)

## -----------------------------------------------------------------------------
rowRanges(ragexp)

## -----------------------------------------------------------------------------
dimnames(ragexp)

## -----------------------------------------------------------------------------
colData(ragexp)

## -----------------------------------------------------------------------------
dim(ragexp)
Reduce(`*`, dim(ragexp))
sparseAssay(ragexp)
length(sparseAssay(ragexp))

## -----------------------------------------------------------------------------
sparseAssay(ragexp, sparse = TRUE)

## -----------------------------------------------------------------------------
compactAssay(ragexp)

## -----------------------------------------------------------------------------
compactAssay(ragexp, sparse = TRUE)

## -----------------------------------------------------------------------------
disjoinAssay(ragexp, simplifyDisjoin = mean)

## -----------------------------------------------------------------------------
unlist(grl, use.names = FALSE)

## -----------------------------------------------------------------------------
rowRanges(ragexp)
assay(ragexp, "score")

## -----------------------------------------------------------------------------
(query <- GRanges(c("chr1:1-14:-", "chr2:11-18:+")))

## -----------------------------------------------------------------------------
weightedmean <- function(scores, ranges, qranges)
{
    isects <- pintersect(ranges, qranges)
    sum(scores * width(isects)) / sum(width(isects))
}

## -----------------------------------------------------------------------------
qreduceAssay(ragexp, query, simplifyReduce = weightedmean)

## -----------------------------------------------------------------------------
library(Matrix)

sm <- Matrix::sparseMatrix(
    i = c(2, 3, 4, 3, 4, 3, 4),
    j = c(1, 1, 1, 3, 3, 4, 4),
    x = c(2L, 4L, 2L, 2L, 2L, 4L, 2L),
    dims = c(4, 4),
    dimnames = list(
        c("chr2:1-10", "chr2:2-10", "chr2:3-10", "chr2:4-10"),
        LETTERS[1:4]
    )
)

as(sm, "RaggedExperiment")

## -----------------------------------------------------------------------------
sessionInfo()

