#include <config.h>
#include <stdio.h>
#include "global.h"
#include "cpu_accel.h"

#ifdef HAVE_ALTIVEC
#define __INCLUDE_FROM_MPEG2ENC_PREDICT_C__
#include "../utils/altivec/altivec_predict.h"
#undef __INCLUDE_FROM_MPEG2ENC_PREDICT_C__
#endif


/* private prototypes */
static void predict_mb (
	pict_data_s *picture,
	uint8_t *oldref[], uint8_t *newref[], uint8_t *cur[],
	int lx, int bx, int by, mbinfo_s *mbi, int secondfield);

static void pred (
	pict_data_s *picture,
	uint8_t *src[], int sfield,
	uint8_t *dst[], int dfield,
	int lx, int w, int h, int x, int y, int dx, int dy, int addflag);

/* static */ void pred_comp (
	pict_data_s *picture,
	uint8_t *src, uint8_t *dst,
	int lx, int w, int h, int x, int y, int dx, int dy, int addflag);

static void calc_DMV 
	(	pict_data_s *picture,int DMV[][2], 
		int *dmvector, int mvx, int mvy);

static void clearblock (pict_data_s *picture,
						uint8_t *cur[], int i0, int j0);
void init_predict(void);


/*
  Initialise prediction - currently purely selection of which
  versions of the various low level computation routines to use
  
  */


static void (*ppred_comp)(
	pict_data_s *picture,
	uint8_t *src, uint8_t *dst,
	int lx, int w, int h, int x, int y, int dx, int dy, int addflag);

void init_predict(void)
{
	int cpucap = cpu_accel();

	if( cpucap  == 0 )	/* No MMX/SSE etc support available */
	{
		ppred_comp = pred_comp;
	}
#ifdef HAVE_ALTIVEC
    else
	{
	    mjpeg_info("SETTING AltiVec for PREDICTION!");
	    ppred_comp = ALTIVEC_SUFFIX(pred_comp);
	}
#endif
}


/* form prediction for a complete picture (frontend for predict_mb)
 *
 * cur:  destination (current) frame
 * secondfield: predict second field of a frame
 * mbi:  macroblock info
 *
 * Notes:
 * - cf. predict_mb
 */

void predict(pict_data_s *picture)
{
	uint8_t **reff = picture->oldref;
	uint8_t **refb = picture->newref;
    uint8_t **pred = picture->pred;
	int secondfield = picture->secondfield;
	int i, j, k;
	mbinfo_s *mbi = picture->mbinfo;
	k = 0;

	/* loop through all macroblocks of the picture */
	for (j=0; j<opt_enc_height; j+=16)
		for (i=0; i<opt_enc_width; i+=16)
		{
			predict_mb(picture,reff,refb,pred,opt_phy_width,i,j,
					   &mbi[k], secondfield );
			k++;
		}
}


static void predict_mb
	( pict_data_s *picture,
	uint8_t *oldref[], uint8_t *newref[], uint8_t *cur[],
	int lx, int bx, int by, mbinfo_s *mbi,  int secondfield )
{
	int addflag;
	int DMV[2][2];

	if (mbi->mb_type&MB_INTRA)
	{
		clearblock(picture,cur,bx,by);
		return;
	}

	addflag = 0; /* first prediction is stored, second is added and averaged */

	if ((mbi->mb_type & MB_FORWARD) || (picture->pict_type==P_TYPE))
	{
		/* forward prediction, including zero MV in P pictures */

		/* frame picture */
		if ((mbi->motion_type==MC_FRAME) || !(mbi->mb_type & MB_FORWARD))
		{
			/* frame-based prediction in frame picture */
			pred(picture,
					oldref,0,cur,0,
					lx,16,16,bx,by,mbi->MV[0][0][0],mbi->MV[0][0][1],0);
		}
		else if (mbi->motion_type==MC_FIELD)
		{
			/* field-based prediction in frame picture
				*
				* note scaling of the vertical coordinates (by, mbi->MV[][0][1])
				* from frame to field!
				*/

			/* top field prediction */
			pred(picture,oldref,mbi->mv_field_sel[0][0],cur,0,
					lx<<1,16,8,bx,by>>1,mbi->MV[0][0][0],mbi->MV[0][0][1]>>1,0);

			/* bottom field prediction */
			pred(picture,oldref,mbi->mv_field_sel[1][0],cur,1,
					lx<<1,16,8,bx,by>>1,mbi->MV[1][0][0],mbi->MV[1][0][1]>>1,0);
		}
		else if (mbi->motion_type==MC_DMV)
		{
			/* dual prime prediction */

			/* calculate derived motion vectors */
			calc_DMV(picture,DMV,mbi->dmvector,mbi->MV[0][0][0],mbi->MV[0][0][1]>>1);

			/* predict top field from top field */
			pred(picture,oldref,0,cur,0,
					lx<<1,16,8,bx,by>>1,mbi->MV[0][0][0],mbi->MV[0][0][1]>>1,0);

			/* predict bottom field from bottom field */
			pred(picture,oldref,1,cur,1,
					lx<<1,16,8,bx,by>>1,mbi->MV[0][0][0],mbi->MV[0][0][1]>>1,0);

			/* predict and add to top field from bottom field */
			pred(picture,oldref,1,cur,0,
					lx<<1,16,8,bx,by>>1,DMV[0][0],DMV[0][1],1);

			/* predict and add to bottom field from top field */
			pred(picture,oldref,0,cur,1,
					lx<<1,16,8,bx,by>>1,DMV[1][0],DMV[1][1],1);
		}
		else
		{
			/* invalid mbi->motion_type in frame picture */
			mjpeg_error_exit1("Internal: invalid motion_type");
		}
		
		addflag = 1; /* next prediction (if any) will be averaged with this one */
	}

	if (mbi->mb_type & MB_BACKWARD)
	{
		/* backward prediction */


		/* frame picture */
		if (mbi->motion_type==MC_FRAME)
		{
			/* frame-based prediction in frame picture */
			pred(picture,newref,0,cur,0,
					lx,16,16,bx,by,mbi->MV[0][1][0],mbi->MV[0][1][1],addflag);
		}
		else
		{
			/* field-based prediction in frame picture
				*
				* note scaling of the vertical coordinates (by, mbi->MV[][1][1])
				* from frame to field!
				*/

			/* top field prediction */
			pred(picture,newref,mbi->mv_field_sel[0][1],cur,0,
					lx<<1,16,8,bx,by>>1,mbi->MV[0][1][0],mbi->MV[0][1][1]>>1,addflag);

			/* bottom field prediction */
			pred(picture,newref,mbi->mv_field_sel[1][1],cur,1,
					lx<<1,16,8,bx,by>>1,mbi->MV[1][1][0],mbi->MV[1][1][1]>>1,addflag);
		}

	}
}

static void pred
	( pict_data_s *picture,
	uint8_t *src[], int sfield,
	uint8_t *dst[], int dfield,
	int lx, int w, int h, int x, int y, int dx, int dy, int addflag )
{
	int cc;

	for (cc=0; cc<3; cc++)
	{
		if (cc==1)
		{
			/* scale for color components */
			if (opt_chroma_format==CHROMA420)
			{
				/* vertical */
				h >>= 1; y >>= 1; dy /= 2;
			}
			if (opt_chroma_format!=CHROMA444)
			{
				/* horizontal */
				w >>= 1; x >>= 1; dx /= 2;
				lx >>= 1;
			}
		}
		(*ppred_comp)(	picture,
						src[cc]+(sfield?lx>>1:0),dst[cc]+(dfield?lx>>1:0),
						lx,w,h,x,y,dx,dy,addflag);
	}
}


/* static */ void pred_comp(
	pict_data_s *picture,
	uint8_t *src,
	uint8_t *dst,
	int lx,
	int w, int h,
	int x, int y,
	int dx, int dy,
	int addflag)
{
	int xint, xh, yint, yh;
	int i, j;
	uint8_t *s, *d;

	/* half pel scaling */
	xint = dx>>1; /* integer part */
	xh = dx & 1;  /* half pel flag */
	yint = dy>>1;
	yh = dy & 1;

	/* origins */
	s = src + lx*(y+yint) + (x+xint); /* motion vector */
	d = dst + lx*y + x;

	if (!xh && !yh)
		if (addflag)
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (unsigned int)(d[i]+s[i]+1)>>1;
				s+= lx;
				d+= lx;
			}
		else
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = s[i];
				s+= lx;
				d+= lx;
			}
	else if (!xh && yh)
		if (addflag)
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (d[i] + ((unsigned int)(s[i]+s[i+lx]+1)>>1)+1)>>1;
				s+= lx;
				d+= lx;
			}
		else
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (unsigned int)(s[i]+s[i+lx]+1)>>1;
				s+= lx;
				d+= lx;
			}
	else if (xh && !yh)
		if (addflag)
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (d[i] + ((unsigned int)(s[i]+s[i+1]+1)>>1)+1)>>1;
				s+= lx;
				d+= lx;
			}
		else
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (unsigned int)(s[i]+s[i+1]+1)>>1;
				s+= lx;
				d+= lx;
			}
	else /* if (xh && yh) */
		if (addflag)
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (d[i] + ((unsigned int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2)+1)>>1;
				s+= lx;
				d+= lx;
			}
		else
			for (j=0; j<h; j++)
			{
				for (i=0; i<w; i++)
					d[i] = (unsigned int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2;
				s+= lx;
				d+= lx;
			}
}


static void calc_DMV
	( pict_data_s *picture,int DMV[][2], 
	int *dmvector, int mvx, int mvy )
{

    if (picture->topfirst)
    {
      /* vector for prediction of top field from bottom field */
      DMV[0][0] = ((mvx  +(mvx>0))>>1) + dmvector[0];
      DMV[0][1] = ((mvy  +(mvy>0))>>1) + dmvector[1] - 1;

      /* vector for prediction of bottom field from top field */
      DMV[1][0] = ((3*mvx+(mvx>0))>>1) + dmvector[0];
      DMV[1][1] = ((3*mvy+(mvy>0))>>1) + dmvector[1] + 1;
    }
    else
    {
      /* vector for prediction of top field from bottom field */
      DMV[0][0] = ((3*mvx+(mvx>0))>>1) + dmvector[0];
      DMV[0][1] = ((3*mvy+(mvy>0))>>1) + dmvector[1] - 1;

      /* vector for prediction of bottom field from top field */
      DMV[1][0] = ((mvx  +(mvx>0))>>1) + dmvector[0];
      DMV[1][1] = ((mvy  +(mvy>0))>>1) + dmvector[1] + 1;
    }

}

static void clearblock
	( pict_data_s *picture,
	uint8_t *cur[], int i0, int j0 )
{
  int i, j, w, h;
  uint8_t *p;

  p = cur[0] 
	  + ((picture->pict_struct==BOTTOM_FIELD) ? opt_phy_width : 0) 
	  + i0 + opt_phy_width2*j0;

  for (j=0; j<16; j++)
  {
    for (i=0; i<16; i++)
      p[i] = 128;
    p+= opt_phy_width2;
  }

  w = h = 16;

  if (opt_chroma_format!=CHROMA444)
  {
    i0>>=1; w>>=1;
  }

  if (opt_chroma_format==CHROMA420)
  {
    j0>>=1; h>>=1;
  }

  p = cur[1] 
	  + ((picture->pict_struct==BOTTOM_FIELD) ? opt_phy_chrom_width : 0) 
	  + i0 + opt_phy_chrom_width2*j0;

  for (j=0; j<h; j++)
  {
    for (i=0; i<w; i++)
      p[i] = 128;
    p+= opt_phy_chrom_width2;
  }

  p = cur[2] 
	  + ((picture->pict_struct==BOTTOM_FIELD) ? opt_phy_chrom_width : 0) 
	  + i0 + opt_phy_chrom_width2*j0;

  for (j=0; j<h; j++)
  {
    for (i=0; i<w; i++)
      p[i] = 128;
    p+= opt_phy_chrom_width2;
  }
}
